#!/bin/bash

currentDir3da=$(pwd)
mhasDir=$currentDir3da/../example
operatingSystem=$(uname -s)
platform=C11
target=$(uname -m)
DebugLevel=1
clobberOn=0
buildCommand="/Rebuild"
executeDecoding=1
checkout=0
vcmTool="vcmSVN"
verbose=0

# loop over arguments
for var in "$@"
do
  if [ "$var" == "Clean" ]; then clobberOn=1; fi
  if [ "$var" == "c89" ]; then platform=C89; fi
  if [ "$var" == "Release" ]; then DebugLevel=0; fi
  if [ "$var" == "nDec" ]; then executeDecoding=0; fi
  if [ "$var" == "co" ]; then checkout=1; fi
  if [ "$var" == "GIT" ]; then vcmTool="vcmGIT"; fi
  if [ "$var" == "v" ]; then verbose=1; fi
done

# set compiler specific options
buildMode=D
OptimLevel=0
clobber=""
if [ $DebugLevel -eq 0 ]; then
    buildMode=R
    OptimLevel=3
fi
if [ $clobberOn -eq 1 ]; then
    clobber="clobber"
    buildCommand="/Clean"
fi
DebugLevelDRC=1
OptimLevelDRC=0
DebugLevelAFsp=0
OptimLevelAFsp=3

# set directories and libtsplite library
if [ $operatingSystem == "Darwin" ]; then
  libtsplite=libtsplite_Mac.a
else
  if [ $target == "x86_64" ]; then
    libtsplite=libtsplite_x86_64.a
  else
    libtsplite=libtsplite.a
  fi
fi
if [ $DebugLevel -eq 0 ]; then
  buildOutDirUnix=$operatingSystem\_$target\_O$OptimLevel
else
  buildOutDirUnix=$operatingSystem\_$target\_$buildMode\_O$OptimLevel
fi
buildOutDirUnixDRC=$operatingSystem\_$target\_D\_O$OptimLevelDRC
buildDir=$currentDir3da/executables_$buildOutDirUnix\_$platform
buildDirAFsp=$currentDir3da/../tools/AFsp

# echo flags and options
DATE=`date +%Y-%m-%d`
echo -e "******************** MPEG-H 3D Audio Coder - Edition 3.0 (CD) *****************"
echo -e "*                                                                             *"
echo -e "*                               Unix OS compiler                              *"
echo -e "*                                                                             *"
echo -e "*                                  $DATE                                 *"
echo -e "*                                                                             *"
echo -e "*    This software may only be used in the development of the MPEG 3D Audio   *"
echo -e "*    standard, ISO/IEC 23008-3 or in conforming implementations or products.  *"
echo -e "*                                                                             *"
echo -e "*******************************************************************************\n"
echo -e "Using $platform compiler [$operatingSystem $target].\n"
echo -e "compile flags CO:   $buildCommand DEBUG=$DebugLevel OPTIM=$OptimLevel $target"
if [ $platform == "C11" ]; then
  echo -e "compile flags HOA:  $buildCommand DEBUG=$DebugLevel OPTIM=$OptimLevel $target"
fi
echo -e "compile flags DRC:  $buildCommand DEBUG=$DebugLevelDRC OPTIM=$OptimLevelDRC $target"
echo -e "compile flags AFsp: $buildCommand DEBUG=$DebugLevelAFsp OPTIM=$OptimLevelAFsp $target\n"

# create output directory
if [ ! -d "$buildDir" ]; then
  mkdir "$buildDir"
else
  rm -rf $buildDir/*
fi
if [ ! -d "$buildDirAFsp/lib" ]; then
  mkdir "$buildDirAFsp/lib"
else
  rm -rf $buildDirAFsp/lib/*.a
fi

# checkout IsoLib
if [ $checkout -eq 1 ]; then
  if [ $vcmTool == "vcmGIT" ]; then
    echo -e "downloading IsoLib from GitHub...\n"
  else
    echo -e "downloading IsoLib from SVN...\n"
  fi
  cd $currentDir3da/../tools/IsoLib/scripts/
  ./checkout.sh -p v $vcmTool > $currentDir3da/coIsoLib.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat coIsoLib.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  fi
  cd $currentDir3da
fi

# check add-on libraries
if [ ! -f $currentDir3da/../tools/AFsp/Makefile ]; then
  echo -e "AFsp is not existing."
  echo -e "Please download http://www.mmsp.ece.mcgill.ca/Documents/Downloads/AFsp/.\n"
  exit 1
fi
if [ ! -f $currentDir3da/../modules/drcCoder/MPEG_D_DRC_refsoft/scripts/compile.sh ]; then
  echo -e "MPEG-D DRC is not existing."
  echo -e "Please add MPEG-D DRC reference software as in ISO/IEC 23003-4.\n"
  exit 1
fi
if [ ! -f $currentDir3da/../tools/IsoLib/libisomediafile/linux/libisomediafile/Makefile ]; then
  echo -e "ISOBMFF is not existing."
  echo -e "Please download https://github.com/MPEGGroup/isobmff/archive/master.zip.\n"
  exit 1
fi

# compile AFsp library
echo Compiling libtsplite
echo Compiling ResampAudio
  make -C  $currentDir3da/../tools/AFsp/ DEBUG=$DebugLevelAFsp OPTIM=$OptimLevelAFsp > libtsplite.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat libtsplite.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $buildDirAFsp/bin/ResampAudio $buildDir/ResampAudio
    mv $buildDirAFsp/lib/libtsplite.a $buildDirAFsp/lib/$libtsplite
  fi

# compile MPEG-H 3D Audio modules
echo Compiling ascParser
  make -C $currentDir3da/../modules/ascParser/ascParserCmdl/make DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > ascParser.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat ascParser.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/ascParser/ascParserCmdl/bin/$buildOutDirUnix/ascParserCmdl $buildDir/ascParserCmdl
  fi

echo Compiling core Encoder
  if [ $clobberOn -eq 1 ] ; then
    make -C $currentDir3da/../modules/coreCoder/mpegD_usac/usacEncDec/make $clobber DEBUG=$DebugLevel OPTIM=$OptimLevel > coreEnc.log 2>&1
  else
    make -C $currentDir3da/../modules/coreCoder/mpegD_usac/usacEncDec/make $clobber usacEnc DEBUG=$DebugLevel OPTIM=$OptimLevel > coreEnc.log 2>&1
  fi
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat coreEnc.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/coreCoder/mpegD_usac/usacEncDec/bin/$buildOutDirUnix\v2_c1_eptool_sbr/usacEnc $buildDir/usacEnc
  fi

echo Compiling core Decoder
  if [ $clobberOn -eq 1 ] ; then
    make -C $currentDir3da/../modules//coreCoder/mpegD_usac/usacEncDec/make $clobber DEBUG=$DebugLevel OPTIM=$OptimLevel > coreDec.log 2>&1
  else
    make -C $currentDir3da/../modules//coreCoder/mpegD_usac/usacEncDec/make $clobber 3DAudioCoreDecoder DEBUG=$DebugLevel OPTIM=$OptimLevel > coreDec.log 2>&1
  fi
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat coreDec.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/coreCoder/mpegD_usac/usacEncDec/bin/$buildOutDirUnix\v2_c1_eptool_sbr/3DAudioCoreDecoder $buildDir/3DAudioCoreDecoder
  fi

echo Compiling Format Converter
  make -C $currentDir3da/../modules/formatConverter/formatConverterCmdl/make DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > formatConverter.log 2>&1

  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat formatConverter.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/formatConverter/formatConverterCmdl/bin/$buildOutDirUnix/formatConverterCmdl $buildDir/formatConverterCmdl
  fi

echo Compiling Mixer
  make -C $currentDir3da/../modules/mixer/mixerCmdl/make/ DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > mixer.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat mixer.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/mixer/mixerCmdl/bin/$buildOutDirUnix/mixerCmdl $buildDir/mixerCmdl
  fi

echo Compiling OAM Encoder
  make -C $currentDir3da/../modules/oamCoder/oamEncoder/oamEncoderCmdl/make DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > oamEncoder.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat oamEncoder.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/oamCoder/oamEncoder/oamEncoderCmdl/bin/$buildOutDirUnix/oamEncoderCmdl $buildDir/oamEncoderCmdl
  fi

echo Compiling OAM Decoder
  make -C $currentDir3da/../modules/oamCoder/oamDecoder/make/ DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > oamDecoder.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat oamDecoder.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/oamCoder/oamDecoder/bin/$buildOutDirUnix/oamDecoder $buildDir/oamDecoder
  fi

echo Compiling gVBAP Renderer
  make -C $currentDir3da/../modules/gVBAPRenderer/gVBAPRendererCmdl/make/ DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > gVBAPRenderer.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat gVBAPRenderer.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/gVBAPRenderer/gVBAPRendererCmdl/bin/$buildOutDirUnix/gVBAPRendererCmdl $buildDir/gVBAPRendererCmdl
  fi

echo Compiling SAOC Decoder
  make -C $currentDir3da/../modules/saocCoder/saocDecoderCmdl/make DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > saocDecoder.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat saocDecoder.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/saocCoder/saocDecoderCmdl/bin/$buildOutDirUnix/saocDecoderCmdl $buildDir/saocDecoderCmdl
  fi

echo Compiling SAOC Encoder
  make -C $currentDir3da/../modules/saocCoder/saocEncoderCmdl/make DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > saocEncoder.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat saocEncoder.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/saocCoder/saocEncoderCmdl/bin/$buildOutDirUnix/saocEncoderCmdl $buildDir/saocEncoderCmdl
  fi

echo Compiling 3D-Audio Decoder
  make -C $currentDir3da/../modules/3DAudioCoder/3DAudioDecoder/make DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > 3DAudioDecoder.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat 3DAudioDecoder.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/3DAudioCoder/3DAudioDecoder/bin/$buildOutDirUnix/3DAudioDecoder $buildDir/3DAudioDecoder
  fi

echo Compiling 3D-Audio Encoder
  make -C $currentDir3da/../modules/3DAudioCoder/3DAudioEncoder/make DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > 3DAudioEncoder.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat 3DAudioEncoder.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/3DAudioCoder/3DAudioEncoder/bin/$buildOutDirUnix/3DAudioEncoder $buildDir/3DAudioEncoder
  fi

echo Compiling dmxMatrixEncoder
  make -C $currentDir3da/../modules/dmxMatrixCoder/dmxMatrixEncoder/dmxMatrixEncCmdl/make  DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > dmxMatrixEncoder.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat dmxMatrixEncoder.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/dmxMatrixCoder/dmxMatrixEncoder/dmxMatrixEncCmdl/bin/$buildOutDirUnix/dmxMatrixEncoderCmdl $buildDir/dmxMatrixEncoderCmdl
  fi

echo Compiling dmxMatrixDecoder
  make -C $currentDir3da/../modules/dmxMatrixCoder/dmxMatrixDecoder/dmxMatrixDecCmdl/make  DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > dmxMatrixDecoder.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat dmxMatrixDecoder.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/dmxMatrixCoder/dmxMatrixDecoder/dmxMatrixDecCmdl/bin/$buildOutDirUnix/dmxMatrixDecoderCmdl $buildDir/dmxMatrixDecoderCmdl
  fi

echo Compiling binauralParametrization_FD
  make -C $currentDir3da/../modules/binaural/parametrization/parametrization_FD/binauralParametrizationCmdl/make DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > binauralParametrization_FD.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat binauralParametrization_FD.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/binaural/parametrization/parametrization_FD/binauralParametrizationCmdl/bin/$buildOutDirUnix/binauralParametrizationFdCmdl $buildDir/binauralParametrizationFdCmdl
  fi

echo Compiling binauralParametrization_TD
  make -C $currentDir3da/../modules/binaural/parametrization/parametrization_TD/binauralParametrizationCmdl/make DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > binauralParametrization_TD.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat binauralParametrization_TD.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/binaural/parametrization/parametrization_TD/binauralParametrizationCmdl/bin/$buildOutDirUnix/binauralParametrizationTdCmdl $buildDir/binauralParametrizationTdCmdl
  fi

echo Compiling binauralRenderer_FD
  make -C $currentDir3da/../modules/binaural/renderer/renderer_FD/binauralRendererCmdl/make DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > binauralRenderer_FD.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat binauralRenderer_FD.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/binaural/renderer/renderer_FD/binauralRendererCmdl/bin/$buildOutDirUnix/binauralRendererFdCmdl $buildDir/binauralRendererFdCmdl
  fi

echo Compiling binauralRenderer_TD
  make -C $currentDir3da/../modules/binaural/renderer/renderer_TD/binauralRendererCmdl/make/ DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > binauralRenderer_TD.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat binauralRenderer_TD.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/binaural/renderer/renderer_TD/binauralRendererCmdl/bin/$buildOutDirUnix/binauralRendererTdCmdl $buildDir/binauralRendererTdCmdl
  fi

echo Compiling binauralBitstreamWriter
  make -C $currentDir3da/../modules/binaural/interface/bitstreamWriterCmdl/make/ DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > binauralBitstreamWriter.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat binauralBitstreamWriter.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/binaural/interface/bitstreamWriterCmdl/bin/$buildOutDirUnix/bitstreamWriterCmdl $buildDir/bitstreamWriterCmdl
  fi

echo Compiling domainSwitcher
  make -C $currentDir3da/../modules/domainSwitcher/domainSwitcherCmdl/make/ DEBUG=$DebugLevel OPTIM=$OptimLevel  $clobber > domainSwitcher.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat domainSwitcher.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/domainSwitcher/domainSwitcherCmdl/bin/$buildOutDirUnix/domainSwitcherCmdl $buildDir/domainSwitcherCmdl
  fi

echo Compiling dynamicObjectPriorityGenerator
  make -C $currentDir3da/../modules/dynamicObjectPriorityGenerator/DynamicObjectPriorityGeneratorCmdl/make/ DEBUG=$DebugLevel OPTIM=$OptimLevel  $clobber > dynamicObjectPriorityGenerator.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat dynamicObjectPriorityGenerator.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/dynamicObjectPriorityGenerator/DynamicObjectPriorityGeneratorCmdl/bin/$buildOutDirUnix/DynamicObjectPriorityGeneratorCmdl $buildDir/DynamicObjectPriorityGeneratorCmdl
  fi

echo Compiling localSetupInformationInterface
  make -C $currentDir3da/../modules/decoderInterfaces/localSetupInformationInterface/localSetupInformationInterfaceExample/make/ DEBUG=$DebugLevel OPTIM=$OptimLevel  $clobber > localSetupInformationInterface.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat localSetupInformationInterface.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../modules/decoderInterfaces/localSetupInformationInterface/localSetupInformationInterfaceExample/bin/$buildOutDirUnix/localSetupInformationInterfaceExample $buildDir/localSetupInformationInterfaceExample
  fi

echo Compiling earconEncoder
    make -C $currentDir3da/../modules/earconCoder/make/ earconEncoder DEBUG=$DebugLevel OPTIM=$OptimLevel  $clobber > earconEncoder.log 2>&1
    if [ $? -ne 0 ]; then
      if [ $verbose -eq 1 ]; then cat earconEncoder.log; fi
      echo "       Build failed. Check corresponding log file for more information."
      exit 1
    else
      cp $currentDir3da/../modules/earconCoder/bin/$buildOutDirUnix/earconEncoder $buildDir/earconEncoder
    fi

echo Compiling earconDecoder
    make -C $currentDir3da/../modules/earconCoder/make/ earconDecoder DEBUG=$DebugLevel OPTIM=$OptimLevel  $clobber > earconDecoder.log 2>&1
    if [ $? -ne 0 ]; then
      if [ $verbose -eq 1 ]; then cat earconDecoder.log; fi
      echo "       Build failed. Check corresponding log file for more information."
      exit 1
    else
      cp $currentDir3da/../modules/earconCoder/bin/$buildOutDirUnix/earconDecoder $buildDir/earconDecoder
    fi

echo Compiling Separator
  make -C $currentDir3da/../tools/separator/separatorCmdl/make/ DEBUG=$DebugLevel OPTIM=$OptimLevel  $clobber > separator.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat separator.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../tools/separator/separatorCmdl/bin/$buildOutDirUnix/separatorCmdl $buildDir/separatorCmdl
  fi

echo Compiling wavM2N
  make -C $currentDir3da/../tools/wavM2N/wavM2NCmdl/make/ DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > wavM2N.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat wavM2N.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../tools/wavM2N/wavM2NCmdl/bin/$buildOutDirUnix/wavM2NCmdl $buildDir/wavM2NCmdl
  fi

echo Compiling cicp2geometry
  make -C $currentDir3da/../tools/cicp2geometry/cicp2geometryCmdl/make/ DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > cicp2geometry.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat cicp2geometry.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../tools/cicp2geometry/cicp2geometryCmdl/bin/$buildOutDirUnix/cicp2geometryCmdl $buildDir/cicp2geometryCmdl
  fi

echo Compiling wavCutter
  make -C $currentDir3da/../tools/wavCutter/wavCutterCmdl/make/ DEBUG=$DebugLevel OPTIM=$OptimLevel  $clobber > wavCutter.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat wavCutter.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../tools/wavCutter/wavCutterCmdl/bin/$buildOutDirUnix/wavCutterCmdl $buildDir/wavCutterCmdl
  fi

echo Compiling prodmetadataEncoder
  make -C $currentDir3da/../tools/prodmetadata/make/ prodmetadataEncoder DEBUG=$DebugLevel OPTIM=$OptimLevel  $clobber > prodmetadataEncoder.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat prodmetadataEncoder.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../tools/prodmetadata/bin/$buildOutDirUnix/prodmetadataEncoder $buildDir/prodmetadataEncoder
  fi

echo Compiling prodmetadataDecoder
  make -C $currentDir3da/../tools/prodmetadata/make/ prodmetadataDecoder DEBUG=$DebugLevel OPTIM=$OptimLevel  $clobber > prodmetadataDecoder.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat prodmetadataDecoder.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../tools/prodmetadata/bin/$buildOutDirUnix/prodmetadataDecoder $buildDir/prodmetadataDecoder
  fi

echo Compiling ssnrcd
  make -C $currentDir3da/../tools/ssnrcd/make/ ssnrcd DEBUG=$DebugLevel OPTIM=$OptimLevel  $clobber > ssnrcd.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat ssnrcd.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDir3da/../tools/ssnrcd/bin/$buildOutDirUnix/ssnrcd $buildDir/ssnrcd
  fi

# compile C11 only projects
if [ $platform == "C11" ]; then
  echo Compiling IarFormatConverter
    make -C $currentDir3da/../modules/iarFormatConverter/iarFormatConverterCmdl/make/ DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > iarFormatConverterCmdl.log 2>&1
    if [ $? -ne 0 ]; then
      if [ $verbose -eq 1 ]; then cat iarFormatConverterCmdl.log; fi
      echo "       Build failed. Check corresponding log file for more information."
      exit 1
    else
      cp $currentDir3da/../modules/iarFormatConverter/iarFormatConverterCmdl/bin/$buildOutDirUnix/iarFormatConverterCmdl $buildDir/iarFormatConverterCmdl
    fi

  echo Compiling HOA Decoder
    make -C $currentDir3da/../modules/hoaCoder/hoaDec/make/ DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > hoaDec.log 2>&1
    if [ $? -ne 0 ]; then
      if [ $verbose -eq 1 ]; then cat hoaDec.log; fi
      echo "       Build failed. Check corresponding log file for more information."
      exit 1
    else
      cp $currentDir3da/../modules/hoaCoder/hoaDec/bin/$buildOutDirUnix/hoaReferenceDecoder $buildDir/hoaReferenceDecoder
    fi

  echo Compiling HOA Encoder
    make -C $currentDir3da/../modules/hoaCoder/hoaEnc/make/ DEBUG=$DebugLevel OPTIM=$OptimLevel $clobber > hoaEnc.log 2>&1
    if [ $? -ne 0 ]; then
      if [ $verbose -eq 1 ]; then cat hoaEnc.log; fi
      echo "       Build failed. Check corresponding log file for more information."
      exit 1
    else
      cp $currentDir3da/../modules/hoaCoder/hoaEnc/bin/$buildOutDirUnix/hoaReferenceEncoder $buildDir/hoaReferenceEncoder
    fi

  echo Compiling hoaMatrixDecCmdl
    make -C $currentDir3da/../modules/hoaMatrixCoder/hoaMatrixDecoder/hoaMatrixDecCmdl/make/ DEBUG=$DebugLevel OPTIM=$OptimLevel  $clobber > hoaMatrixDecCmdl.log 2>&1
    if [ $? -ne 0 ]; then
      if [ $verbose -eq 1 ]; then cat hoaMatrixDecCmdl.log; fi
      echo "       Build failed. Check corresponding log file for more information."
      exit 1
    else
      cp $currentDir3da/../modules/hoaMatrixCoder/hoaMatrixDecoder/hoaMatrixDecCmdl/bin/$buildOutDirUnix/hoaMatrixDecoderCmdl $buildDir/hoaMatrixDecoderCmdl
    fi

    echo Compiling hoaMatrixEncCmdl
    make -C $currentDir3da/../modules/hoaMatrixCoder/hoaMatrixEncoder/hoaMatrixEncCmdl/make/ DEBUG=$DebugLevel OPTIM=$OptimLevel  $clobber > hoaMatrixEncCmdl.log 2>&1
    if [ $? -ne 0 ]; then
      if [ $verbose -eq 1 ]; then cat hoaMatrixEncCmdl.log; fi
      echo "       Build failed. Check corresponding log file for more information."
      exit 1
    else
      cp $currentDir3da/../modules/hoaMatrixCoder/hoaMatrixEncoder/hoaMatrixEncCmdl/bin/$buildOutDirUnix/hoaMatrixEncoderCmdl $buildDir/hoaMatrixEncoderCmdl
    fi
fi

# compile DRC decoder
echo Compiling drcCoder
  chmod -R u+rwx $currentDir3da/../modules/drcCoder/MPEG_D_DRC_refsoft
  cd $currentDir3da/../modules/drcCoder/MPEG_D_DRC_refsoft/scripts/
  currentDirDrc=$currentDir3da/../modules/drcCoder/MPEG_D_DRC_refsoft/scripts/

  ./compile.sh MPEGH $clobber > DrcTool.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat DrcTool.log; fi
    echo "       Build failed. Check corresponding log file for more information."
    exit 1
  else
    cp $currentDirDrc/../modules/drcTool/drcToolDecoder/bin/$buildOutDirUnixDRC/drcToolDecoder $buildDir/drcToolDecoder
    cp $currentDirDrc/../modules/drcTool/drcToolEncoder/bin/$buildOutDirUnixDRC/drcToolEncoder $buildDir/drcToolEncoder
    cp $currentDirDrc/../modules/loudnessNormalizer/loudnessNormalizerCmdl/bin/$buildOutDirUnixDRC/loudnessNormalizerCmdl $buildDir/loudnessNormalizerCmdl
    cp $currentDirDrc/../modules/peakLimiter/peakLimiterCmdl/bin/$buildOutDirUnixDRC/peakLimiterCmdl $buildDir/peakLimiterCmdl
    cp $currentDirDrc/../modules/uniDrcModules/uniDrcBitstreamDecoderCmdl/bin/$buildOutDirUnixDRC/uniDrcBitstreamDecoderCmdl $buildDir/uniDrcBitstreamDecoderCmdl
    cp $currentDirDrc/../modules/uniDrcModules/uniDrcGainDecoderCmdl/bin/$buildOutDirUnixDRC/uniDrcGainDecoderCmdl $buildDir/uniDrcGainDecoderCmdl
    cp $currentDirDrc/../modules/uniDrcModules/uniDrcInterfaceDecoderCmdl/bin/$buildOutDirUnixDRC/uniDrcInterfaceDecoderCmdl $buildDir/uniDrcInterfaceDecoderCmdl
    cp $currentDirDrc/../modules/uniDrcModules/uniDrcInterfaceEncoderCmdl/bin/$buildOutDirUnixDRC/uniDrcInterfaceEncoderCmdl $buildDir/uniDrcInterfaceEncoderCmdl
    cp $currentDirDrc/../modules/uniDrcModules/uniDrcSelectionProcessCmdl/bin/$buildOutDirUnixDRC/uniDrcSelectionProcessCmdl $buildDir/uniDrcSelectionProcessCmdl
    cp $currentDirDrc/../tools/dmx/dmxCmdl/bin/$buildOutDirUnixDRC/dmxCmdl $buildDir/dmxCmdl
    cp $currentDirDrc/../tools/qmflib/make/bin/$buildOutDirUnixDRC/qmflib_example $buildDir/qmflib_example
    cp $currentDirDrc/../tools/fftlib/make/bin/$buildOutDirUnixDRC/fftlib_example $buildDir/fftlib_example
  fi

# perform example decoding of mhas bitstreams
if [ $executeDecoding -eq 1 ]; then
  cd $currentDir3da
  echo -e "\nDecoding example__C2_3_FD.mhas"
  $buildDir/3DAudioDecoder -if "$mhasDir/example__C2_3_FD.mhas" -of "$mhasDir/example__C2_3_FD.wav" > decoding.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat decoding.log; fi
    echo "       Decoding failed. Check corresponding log file for more information."
    exit 1
  fi

  echo -e "Decoding example__C6_3_FD.mhas"
  $buildDir/3DAudioDecoder -if "$mhasDir/example__C6_3_FD.mhas" -of "$mhasDir/example__C6_3_FD.wav" >> decoding.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat decoding.log; fi
    echo "       Decoding failed. Check corresponding log file for more information."
    exit 1
  fi

  echo -e "Decoding example__C19_3_FD.mhas"
  $buildDir/3DAudioDecoder -if "$mhasDir/example__C19_3_FD.mhas" -of "$mhasDir/example__C19_3_FD.wav" >> decoding.log 2>&1
  if [ $? -ne 0 ]; then
    if [ $verbose -eq 1 ]; then cat decoding.log; fi
    echo "       Decoding failed. Check corresponding log file for more information."
    exit 1
  fi
fi

echo -e "\nAll executables are collected in:"
echo -e $buildDir/
if [ $executeDecoding -eq 1 ]; then
  echo -e "\nAll decoded .wav files are collected in:"
  echo -e $mhasDir/
fi
echo -e "\n       Build successful."
exit 0
