#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include "signal_classifier.h"

CLASSIFICATION classfyData;

#define FRAMELEN	1024
#define MIN_POW		-200
#define LOWFREQUENCY	7500	/* the upper limit of low-frequency domain(Hz) */
#define INDEXOFLOWFREQUENCY 160 /* ((int)(LOWFREQUENCY*512/24000)) */

#define NFRAMEAHEAD	1
#define AVE_TONAL_LENGTH	       100	/* the length of the long-term AVE */
#define AVE_TONAL_LENGTH_SHORT		10	/* the length of the short-term AVE */
#define SPECTRAL_TILT_LENGTH		80	/* the length of the long-term MSD */
#define SPECTRAL_TILT_LENGTH_SHORT	20	/* the length of the short-term MSD */
#define SMOOTHING_LENGTH	       100      /* the length of smoothing */

#define NO_BORDER			0
#define BORDER_MUSIC_SPEECH		1
#define BORDER_MUSIC_SPEECH_DEFINITE	2
#define BORDER_SPEECH_MUSIC		3
#define BORDER_SPEECH_MUSIC_DEFINITE	4

#define TBD				0
#define SPEECH_DEFINITE		        1
#define SPEECH				2
#define MUSIC_DEFINITE		        3
#define MUSIC				4

#define PI		3.1415926535897932f

#ifndef max
#define max(a, b) ((a) > (b) ? (a) : (b))
#endif

#ifndef min
#define min(a, b) ((a) < (b) ? (a) : (b))
#endif

/* hanning window */
const double hanning_window[FRAMELEN]=
{
	0,                0.000015370317393,0.000061480690889,0.000138329384457,0.000245913504789,
	0.000384229001402,0.000553270666797,0.00075303213665, 0.000983505890054,0.001244683249805,
	0.00153655438272, 0.001859108300018,0.002212332857725,0.002596214757137,0.003010739545316,
	0.00345589161564, 0.003931654208384,0.004438009411355,0.004974938160566,0.005542420240954,
	0.006140434287139,0.006768957784229,0.007427967068671,0.008117437329137,0.008837342607462,
	0.009587655799618,0.010368348656739,0.01117939178618, 0.012020754652625,0.012892405579237,
	0.013794311748852,0.014726439205213,0.015688752854247,0.01668121646539, 0.017703792672947,
	0.018756442977503,0.019839127747368,0.020951806220072,0.022094436503901,0.023266975579469,
	0.024469379301344,0.025701602399704,0.026963598482046,0.028255320034932,0.029576718425774,
	0.030927743904671,0.032308345606277,0.033718471551717,0.035158068650547,0.03662708270275,
	0.038125458400778,0.039653139331631,0.041210067978987,0.042796185725362,0.044411432854319,
	0.046055748552716,0.047729070912998,0.049431336935522,0.051162482530936,0.052922442522585,
	0.054711150648972,0.056528539566246,0.058374540850741,0.060249085001554,0.062152101443155,
	0.064083518528051,0.066043263539481,0.068031262694153,0.070047441145022,0.072091722984109,
	0.074164031245358,0.076264287907535,0.078392413897163,0.080548329091501,0.082731952321562,
	0.084943201375164,0.08718199300003, 0.089448242906922,0.091741865772811,0.094062775244093,
	0.096410883939837,0.098786103455079,0.101188344364146,0.103617516224025,0.10607352757777,
	0.108556285957941,0.111065697890088,0.11360166889627, 0.116164103498613,0.118752905222901,
	0.121367976602213,0.12400921918059, 0.126676533516741,0.129369819187789,0.132088974793051,
	0.134833897957855,0.137604485337396,0.140400632620624,0.143222234534175,0.146069184846332,
	0.148941376371024,0.151838700971864,0.154761049566218,0.157708312129314,0.160680377698381,
	0.16367713437683, 0.166698469338468,0.169744268831739,0.172814418184015,0.175908801805908,
	0.179027303195622,0.182169804943345,0.18533618873566, 0.188526335360008,0.191740124709171,
	0.194977435785797,0.198238146706954,0.201522134708718,0.204829276150797,0.208159446521186,
	0.211512520440851,0.214888371668455,0.218286873105108,0.22170789679915, 0.225151313950974,
	0.22861699491787, 0.232104809218908,0.235614625539852,0.2391463117381,  0.242699734847664,
	0.246274761084171,0.249871255849904,0.253489083738869,0.25712810854189, 0.26078819325174,
	0.264469200068298,0.268170990403738,0.271893424887747,0.27563636337277, 0.279399664939288,
	0.283183187901125,0.286986789810779,0.290810327464789,0.294653656909122,0.298516633444598,
	0.302399111632333,0.306300945299219,0.310221987543421,0.314162090739918,0.318121106546049,
	0.322098885907107,0.326095279061949,0.33011013554863, 0.334143304210072,0.338194633199756,
	0.342263969987435,0.346351161364878,0.35045605345164, 0.354578491700855,0.358718320905051,
	0.362875385202001,0.367049528080581,0.371240592386673,0.375448420329074,0.37967285348544,
	0.38391373280825, 0.388170898630795,0.392444190673188,0.396733448048399,0.401038509268312,
	0.405359212249804,0.409695394320852,0.41404689222665, 0.418413542135763,0.422795179646288,
	0.427191639792051,0.431602757048812,0.436028365340499,0.440468298045461,0.444922388002741,
	0.449390467518371,0.453872368371681,0.458367921821638,0.462876958613196,0.467399308983666,
	0.471934802669115,0.476483268910768,0.481044536461443,0.485618433591993,0.490204788097777,
	0.494803427305141,0.499414178077916,0.504036866823943,0.508671319501604,0.513317361626373,
	0.517974818277392,0.522643514104049,0.527323273332586,0.532013919772712,0.53671527682424,
	0.541427167483736,0.546149414351177,0.550881839636639,0.555624265166984,0.560376512392572,
	0.565138402393982,0.569909755888744,0.574690393238098,0.579480134453749,0.584278799204648,
	0.589086206823778,0.59390217631496, 0.598726526359666,0.603559075323841,0.60839964126475,
	0.61324804193782, 0.618104094803507,0.622967617034165,0.62783842552093, 0.632716336880616,
	0.637601167462617,0.642492733355824,0.647390850395544,0.652295334170439,0.657206000029468,
	0.662122663088837,0.667045138238959,0.671973240151429,0.676906783285994,0.681845581897543,
	0.686789450043102,0.691738201588827,0.69669165021702, 0.701649609433141,0.706611892572826,
	0.711578312808922,0.716548683158512,0.721522816489962,0.726500525529963,0.731481622870585,
	0.736465920976327,0.741453232191182,0.746443368745703,0.751436142764068,0.756431366271157,
	0.76142885119963, 0.766428409397002,0.771429852632734,0.776432992605314,0.78143764094935,
	0.786443609242662,0.791450709013371,0.796458751747002,0.801467548893576,0.806476911874712,
	0.811486652090724,0.816496580927726,0.821506509764728,0.82651624998074, 0.831525612961876,
	0.83653441010845, 0.841542452842081,0.84654955261279, 0.851555520906101,0.856560169250138,
	0.861563309222718,0.86656475245845, 0.871564310655822,0.876561795584295,0.881557019091384,
	0.886549793109749,0.89153992966427, 0.896527240879125,0.901511538984867,0.906492636325489,
	0.91147034536549, 0.91644447869694, 0.92141484904653, 0.926381269282625,0.931343552422311,
	0.936301511638432,0.941254960266625,0.94620371181235, 0.951147579957909,0.956086378569458,
	0.961019921704023,0.965948023616493,0.970870498766615,0.975787161825984,0.980697827685013,
	0.985602311459908,0.990500428499628,0.995391994392835,1.000276824974836,1.005154736334522,
	1.010025544821287,1.014889067051945,1.019745119917632,1.024593520590702,1.029434086531611,
	1.034266635495786,1.039090985540492,1.043906955031674,1.048714362650804,1.053513027401703,
	1.058302768617354,1.063083405966708,1.06785475946147, 1.07261664946288, 1.077368896688468,
	1.082111322218813,1.086843747504275,1.091565994371717,1.096277885031211,1.10097924208274,
	1.105669888522866,1.110349647751403,1.11501834357806, 1.119675800229079,1.124321842353848,
	1.128956295031508,1.133578983777536,1.138189734550311,1.142788373757675,1.147374728263459,
	1.151948625394009,1.156509892944684,1.161058359186337,1.165593852871786,1.170116203242256,
	1.174625240033814,1.179120793483771,1.183602694337081,1.18807077385271, 1.192524863809992,
	1.196964796514953,1.20139040480664, 1.205801522063401,1.210197982209164,1.214579619719689,
	1.218946269628802,1.2232977675346,  1.227633949605648,1.23195465258714, 1.236259713807053,
	1.240548971182264,1.244822263224656,1.249079429047202,1.253320308370012,1.257544741526378,
	1.261752569468779,1.265943633774871,1.270117776653451,1.274274840950401,1.278414670154597,
	1.282537108403812,1.286642000490574,1.290729191868017,1.294798528655696,1.298849857645379,
	1.302883026306822,1.306897882793503,1.310894275948344,1.314872055309404,1.318831071115534,
	1.322771174312031,1.326692216556233,1.330594050223119,1.334476528410854,1.33833950494633,
	1.342182834390663,1.346006372044673,1.349809973954327,1.353593496916164,1.357356798482682,
	1.361099736967705,1.364822171451714,1.368523961787154,1.372204968603712,1.375865053313562,
	1.379504078116583,1.383121906005548,1.386718400771281,1.390293427007788,1.393846850117352,
	1.3973785363156,  1.400888352636544,1.404376166937582,1.407841847904478,1.411285265056302,
	1.414706288750344,1.418104790186997,1.421480641414601,1.424833715334266,1.428163885704655,
	1.431471027146734,1.434755015148498,1.438015726069655,1.441253037146281,1.444466826495444,
	1.447656973119792,1.450823356912107,1.45396585865983, 1.457084360049544,1.460178743671437,
	1.463248893023713,1.466294692516984,1.469316027478622,1.472312784157071,1.475284849726139,
	1.478232112289234,1.481154460883588,1.484051785484428,1.48692397700912, 1.489770927321276,
	1.492592529234828,1.495388676518056,1.498159263897597,1.500904187062401,1.503623342667663,
	1.506316628338712,1.508983942674862,1.511625185253239,1.514240256632551,1.516829058356839,
	1.519391492959181,1.521927463965364,1.524436875897511,1.526919634277682,1.529375645631426,
	1.531804817491306,1.534207058400374,1.536582277915615,1.53893038661136, 1.541251296082641,
	1.54354491894853, 1.545811168855422,1.548049960480288,1.55026120953389, 1.552444832763951,
	1.55460074795829, 1.556728873947918,1.558829130610094,1.560901438871343,1.56294572071043,
	1.5649618991613,  1.566949898315971,1.568909643327401,1.570841060412298,1.572744076853898,
	1.574618621004711,1.576464622289206,1.57828201120648,1.580070719332867, 1.581830679324516,
	1.58356182491993, 1.585264090942454,1.586937413302736,1.588581729001133,1.59019697613009,
	1.591783093876465,1.593340022523821,1.594867703454674,1.596366079152702,1.597835093204905,
	1.599274690303735,1.600684816249176,1.602065417950781,1.603416443429678,1.60473784182052,
	1.606029563373406,1.607291559455748,1.608523782554108,1.609726186275983,1.610898725351551,
	1.612041355635379,1.613154034108084,1.614236718877949,1.615289369182504,1.616311945390062,
	1.617304409001205,1.618266722650239,1.6191988501066,  1.620100756276215,1.620972407202827,
	1.621813770069272,1.622624813198712,1.623405506055834,1.624155819247991,1.624875724526315,
	1.625565194786781,1.626224204071223,1.626852727568314,1.627450741614498,1.628018223694886,
	1.628555152444097,1.629061507647068,1.629537270239812,1.629982422310136,1.630396947098315,
	1.630780828997727,1.631134053555434,1.631456607472732,1.631748478605648,1.632009655965398,
	1.632240129718802,1.632439891188656,1.63260893285405, 1.632747248350663,1.632854832470995,
	1.632931681164564,1.632977791538059,1.632993161855452,1.632977791538059,1.632931681164564,
	1.632854832470995,1.632747248350663,1.63260893285405, 1.632439891188656,1.632240129718802,
	1.632009655965398,1.631748478605648,1.631456607472732,1.631134053555434,1.630780828997727,
	1.630396947098315,1.629982422310136,1.629537270239812,1.629061507647068,1.628555152444097,
	1.628018223694886,1.627450741614498,1.626852727568314,1.626224204071223,1.625565194786781,
	1.624875724526315,1.624155819247991,1.623405506055834,1.622624813198712,1.621813770069272,
	1.620972407202827,1.620100756276215,1.6191988501066,  1.618266722650239,1.617304409001205,
	1.616311945390062,1.615289369182504,1.614236718877949,1.613154034108084,1.612041355635379,
	1.610898725351551,1.609726186275983,1.608523782554108,1.607291559455748,1.606029563373406,
	1.60473784182052, 1.603416443429678,1.602065417950781,1.600684816249176,1.599274690303735,
	1.597835093204905,1.596366079152702,1.594867703454674,1.593340022523821,1.591783093876465,
	1.59019697613009, 1.588581729001133,1.586937413302736,1.585264090942454,1.58356182491993,
	1.581830679324516,1.580070719332867,1.57828201120648, 1.576464622289206,1.574618621004711,
	1.572744076853898,1.570841060412298,1.568909643327401,1.566949898315971,1.5649618991613,
	1.56294572071043, 1.560901438871343,1.558829130610094,1.556728873947918,1.55460074795829,
	1.552444832763951,1.55026120953389, 1.548049960480288,1.545811168855422,1.54354491894853,
	1.541251296082641,1.53893038661136, 1.536582277915615,1.534207058400374,1.531804817491306,
	1.529375645631426,1.526919634277682,1.524436875897511,1.521927463965364,1.519391492959181,
	1.516829058356839,1.514240256632551,1.511625185253239,1.508983942674862,1.506316628338712,
	1.503623342667663,1.500904187062401,1.498159263897597,1.495388676518056,1.492592529234828,
	1.489770927321276,1.48692397700912, 1.484051785484428,1.481154460883588,1.478232112289234,
	1.475284849726139,1.472312784157071,1.469316027478622,1.466294692516984,1.463248893023713,
	1.460178743671437,1.457084360049544,1.45396585865983, 1.450823356912107,1.447656973119792,
	1.444466826495444,1.441253037146281,1.438015726069655,1.434755015148498,1.431471027146734,
	1.428163885704655,1.424833715334266,1.421480641414601,1.418104790186997,1.414706288750344,
	1.411285265056302,1.407841847904478,1.404376166937582,1.400888352636544,1.3973785363156,
	1.393846850117352,1.390293427007788,1.386718400771281,1.383121906005548,1.379504078116583,
	1.375865053313562,1.372204968603712,1.368523961787154,1.364822171451714,1.361099736967705,
	1.357356798482682,1.353593496916164,1.349809973954327,1.346006372044673,1.342182834390663,
	1.33833950494633, 1.334476528410854,1.330594050223119,1.326692216556233,1.322771174312031,
	1.318831071115534,1.314872055309404,1.310894275948344,1.306897882793503,1.302883026306822,
	1.298849857645379,1.294798528655696,1.290729191868017,1.286642000490574,1.282537108403812,
	1.278414670154597,1.274274840950401,1.270117776653451,1.265943633774871,1.261752569468779,
	1.257544741526378,1.253320308370012,1.249079429047202,1.244822263224656,1.240548971182264,
	1.236259713807053,1.23195465258714, 1.227633949605648,1.2232977675346,  1.218946269628802,
	1.214579619719689,1.210197982209164,1.205801522063401,1.20139040480664, 1.196964796514953,
	1.192524863809992,1.18807077385271, 1.183602694337081,1.179120793483771,1.174625240033814,
	1.170116203242256,1.165593852871786,1.161058359186337,1.156509892944684,1.151948625394009,
	1.147374728263459,1.142788373757675,1.138189734550311,1.133578983777536,1.128956295031508,
	1.124321842353848,1.119675800229079,1.11501834357806, 1.110349647751403,1.105669888522866,
	1.10097924208274, 1.096277885031211,1.091565994371717,1.086843747504275,1.082111322218813,
	1.077368896688468,1.07261664946288, 1.06785475946147, 1.063083405966708,1.058302768617354,
	1.053513027401703,1.048714362650804,1.043906955031674,1.039090985540492,1.034266635495786,
	1.029434086531611,1.024593520590702,1.019745119917632,1.014889067051945,1.010025544821287,
	1.005154736334522,1.000276824974836,0.995391994392835,0.990500428499628,0.985602311459908,
	0.980697827685013,0.975787161825984,0.970870498766615,0.965948023616493,0.961019921704023,
	0.956086378569458,0.951147579957909,0.94620371181235, 0.941254960266625,0.936301511638432,
	0.931343552422311,0.926381269282625,0.92141484904653, 0.91644447869694, 0.91147034536549,
	0.906492636325489,0.901511538984867,0.896527240879125,0.89153992966427, 0.886549793109749,
	0.881557019091384,0.876561795584295,0.871564310655822,0.86656475245845, 0.861563309222718,
	0.856560169250138,0.851555520906101,0.84654955261279, 0.841542452842081,0.83653441010845,
	0.831525612961876,0.82651624998074, 0.821506509764728,0.816496580927726,0.811486652090724,
	0.806476911874712,0.801467548893576,0.796458751747002,0.791450709013371,0.786443609242662,
	0.78143764094935, 0.776432992605314,0.771429852632734,0.766428409397002,0.76142885119963,
	0.756431366271157,0.751436142764068,0.746443368745703,0.741453232191182,0.736465920976327,
	0.731481622870585,0.726500525529963,0.721522816489962,0.716548683158512,0.711578312808922,
	0.706611892572826,0.701649609433141,0.69669165021702, 0.691738201588827,0.686789450043102,
	0.681845581897543,0.676906783285994,0.671973240151429,0.667045138238959,0.662122663088837,
	0.657206000029468,0.652295334170439,0.647390850395544,0.642492733355824,0.637601167462617,
	0.632716336880616,0.62783842552093, 0.622967617034165,0.618104094803507,0.61324804193782,
	0.60839964126475, 0.603559075323841,0.598726526359666,0.59390217631496, 0.589086206823778,
	0.584278799204648,0.579480134453749,0.574690393238098,0.569909755888744,0.565138402393982,
	0.560376512392572,0.555624265166984,0.550881839636639,0.546149414351177,0.541427167483736,
	0.53671527682424,0.532013919772712,0.527323273332586,0.522643514104049,0.517974818277392,
	0.513317361626373,0.508671319501604,0.504036866823943,0.499414178077916,0.494803427305141,
	0.490204788097777,0.485618433591993,0.481044536461443,0.476483268910768,0.471934802669115,
	0.467399308983666,0.462876958613196,0.458367921821638,0.453872368371681,0.449390467518371,
	0.444922388002741,0.440468298045461,0.436028365340499,0.431602757048812,0.427191639792051,
	0.422795179646288,0.418413542135763,0.41404689222665, 0.409695394320852,0.405359212249804,
	0.401038509268312,0.396733448048399,0.392444190673188,0.388170898630795,0.38391373280825,
	0.37967285348544, 0.375448420329074,0.371240592386673,0.367049528080581,0.362875385202001,
	0.358718320905051,0.354578491700855,0.35045605345164, 0.346351161364878,0.342263969987435,
	0.338194633199756,0.334143304210072,0.33011013554863, 0.326095279061949,0.322098885907107,
	0.318121106546049,0.314162090739918,0.310221987543421,0.306300945299219,0.302399111632333,
	0.298516633444598,0.294653656909122,0.290810327464789,0.286986789810779,0.283183187901125,
	0.279399664939288,0.27563636337277, 0.271893424887747,0.268170990403738,0.264469200068298,
	0.26078819325174, 0.25712810854189, 0.253489083738869,0.249871255849904,0.246274761084171,
	0.242699734847664,0.2391463117381,  0.235614625539852,0.232104809218908,0.22861699491787,
	0.225151313950974,0.22170789679915, 0.218286873105108,0.214888371668455,0.211512520440851,
	0.208159446521186,0.204829276150797,0.201522134708718,0.198238146706954,0.194977435785797,
	0.191740124709171,0.188526335360008,0.18533618873566, 0.182169804943345,0.179027303195622,
	0.175908801805908,0.172814418184015,0.169744268831739,0.166698469338468,0.16367713437683,
	0.160680377698381,0.157708312129314,0.154761049566218,0.151838700971864,0.148941376371024,
	0.146069184846332,0.143222234534175,0.140400632620624,0.137604485337396,0.134833897957855,
	0.132088974793051,0.129369819187789,0.126676533516741,0.12400921918059, 0.121367976602213,
	0.118752905222901,0.116164103498613,0.11360166889627, 0.111065697890088,0.108556285957941,
	0.10607352757777, 0.103617516224025,0.101188344364146,0.098786103455079,0.096410883939837,
	0.094062775244093,0.091741865772811,0.089448242906922,0.08718199300003, 0.084943201375164,
	0.082731952321562,0.080548329091501,0.078392413897163,0.076264287907535,0.074164031245358,
	0.072091722984109,0.070047441145022,0.068031262694153,0.066043263539481,0.064083518528051,
	0.062152101443155,0.060249085001554,0.058374540850741,0.056528539566246,0.054711150648972,
	0.052922442522585,0.051162482530936,0.049431336935522,0.047729070912998,0.046055748552716,
	0.044411432854319,0.042796185725362,0.041210067978987,0.039653139331631,0.038125458400778,
	0.03662708270275, 0.035158068650547,0.033718471551717,0.032308345606277,0.030927743904671,
	0.029576718425774,0.028255320034932,0.026963598482046,0.025701602399704,0.024469379301344,
	0.023266975579469,0.022094436503901,0.020951806220072,0.019839127747368,0.018756442977503,
	0.017703792672947,0.01668121646539, 0.015688752854247,0.014726439205213,0.013794311748852,
	0.012892405579237,0.012020754652625,0.01117939178618, 0.010368348656739,0.009587655799618,
	0.008837342607462,0.008117437329137,0.007427967068671,0.006768957784229,0.006140434287139,
	0.005542420240954,0.004974938160566,0.004438009411355,0.003931654208384,0.00345589161564,
	0.003010739545316,0.002596214757137,0.002212332857725,0.001859108300018,0.00153655438272,
	0.001244683249805,0.000983505890054,0.00075303213665,0.000553270666797,0.000384229001402,
	0.000245913504789,0.000138329384457,0.000061480690889,0.000015370317393,
};

const double absolute_threshold[FRAMELEN/2]=
{
	42.09507, 24.17017, 17.46814, 13.87051, 11.59598, 10.01474, 8.84462, 7.93937, 
	7.21520, 6.62048, 6.12152, 5.69534, 5.32567, 5.00063, 4.71133, 4.45091, 
	4.21402, 3.99635, 3.79442, 3.60533, 3.42667, 3.25636, 3.09262, 2.93390, 
	2.77881, 2.62613, 2.47477, 2.32374, 2.17215, 2.01920, 1.86415, 1.70637, 
	1.54526, 1.38033, 1.21115, 1.03736, 0.85867, 0.67489, 0.48590, 0.29165, 
	0.09220, -0.11230, -0.32164, -0.53548, -0.75341, -0.97493, -1.19943, -1.42623, 
	-1.65456, -1.88357, -2.11233, -2.33984, -2.56507, -2.78692, -3.00424, -3.21588, 
	-3.42065, -3.61737, -3.80489, -3.98204, -4.14774, -4.30091, -4.44058, -4.56582, 
	-4.67582, -4.76986, -4.84730, -4.90767, -4.95058, -4.97579, -4.98318, -4.97278, 
	-4.94475, -4.89936, -4.83705, -4.75833, -4.66388, -4.55446, -4.43092, -4.29421, 
	-4.14537, -3.98547, -3.81566, -3.63712, -3.45104, -3.25863, -3.06111, -2.85966, 
	-2.65545, -2.44962, -2.24324, -2.03734, -1.83289, -1.63077, -1.43181, -1.23673, 
	-1.04621, -0.86080, -0.68101, -0.50722, -0.33977, -0.17890, -0.02478, 0.12250, 
	0.26289, 0.39643, 0.52321, 0.64336, 0.75706, 0.86454, 0.96603, 1.06183, 
	1.15223, 1.23756, 1.31812, 1.39428, 1.46636, 1.53471, 1.59966, 1.66154, 
	1.72068, 1.77738, 1.83195, 1.88466, 1.93580, 1.98561, 2.03434, 2.08222, 
	2.12944, 2.17621, 2.22270, 2.26908, 2.31550, 2.36211, 2.40902, 2.45635, 
	2.50420, 2.55268, 2.60186, 2.65183, 2.70265, 2.75439, 2.80711, 2.86085, 
	2.91567, 2.97161, 3.02871, 3.08700, 3.14652, 3.20730, 3.26937, 3.33276, 
	3.39749, 3.46358, 3.53107, 3.59996, 3.67029, 3.74208, 3.81534, 3.89009, 
	3.96636, 4.04417, 4.12353, 4.20446, 4.28698, 4.37112, 4.45689, 4.54431, 
	4.63340, 4.72417, 4.81666, 4.91087, 5.00683, 5.10455, 5.20407, 5.30538, 
	5.40852, 5.51351, 5.62037, 5.72911, 5.83975, 5.95233, 6.06685, 6.18334, 
	6.30182, 6.42230, 6.54482, 6.66940, 6.79604, 6.92478, 7.05564, 7.18864, 
	7.32379, 7.46113, 7.60067, 7.74244, 7.88646, 8.03275, 8.18133, 8.33223, 
	8.48547, 8.64107, 8.79906, 8.95945, 9.12228, 9.28756, 9.45532, 9.62558, 
	9.79837, 9.97371, 10.15162, 10.33214, 10.51527, 10.70105, 10.88950, 11.08065, 
	11.27452, 11.47113, 11.67052, 11.87270, 12.07770, 12.28554, 12.49626, 12.70988, 
	12.92642, 13.14591, 13.36837, 13.59383, 13.82232, 14.05386, 14.28849, 14.52622, 
	14.76708, 15.01110, 15.25831, 15.50873, 15.76239, 16.01932, 16.27954, 16.54309, 
	16.80999, 17.08026, 17.35394, 17.63106, 17.91164, 18.19570, 18.48329, 18.77442, 
	19.06913, 19.36744, 19.66938, 19.97499, 20.28428, 20.59730, 20.91406, 21.23460, 
	21.55896, 21.88714, 22.21920, 22.55515, 22.89503, 23.23887, 23.58669, 23.93853, 
	24.29442, 24.65439, 25.01846, 25.38668, 25.75907, 26.13566, 26.51648, 26.90157, 
	27.29095, 27.68466, 28.08272, 28.48518, 28.89206, 29.30340, 29.71922, 30.13956, 
	30.56445, 30.99392, 31.42801, 31.86674, 32.31016, 32.75829, 33.21117, 33.66883, 
	34.13130, 34.59862, 35.07082, 35.54793, 36.02999, 36.51703, 37.00909, 37.50620, 
	38.00839, 38.51569, 39.02815, 39.54580, 40.06866, 40.59679, 41.13020, 41.66894, 
	42.21304, 42.76254, 43.31746, 43.87786, 44.44376, 45.01519, 45.59220, 46.17482, 
	46.76309, 47.35704, 47.95670, 48.56212, 49.17334, 49.79038, 50.41328, 51.04209, 
	51.67683, 52.31755, 52.96429, 53.61707, 54.27594, 54.94094, 55.61210, 56.28946, 
	56.97306, 57.66293, 58.35912, 59.06166, 59.77059, 60.48596, 61.20779, 61.93613, 
	62.67101, 63.41248, 64.16057, 64.91533, 65.67678, 66.44498, 67.21997, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 
	68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177, 68.00177
};

/*===================================================================================
 * Function: complex_FFT
 *		The conventional complex FFT without speeding techniques.
 *      For specific DSP, they may have special FFT routines.
 * Arguments:
 *		float *x:		Input real data array
 *		float *y:		Input image data array
 *		int n:			The size of FFT = log2(N)
 *		int direction:	+1 = forward FFT, -1 = inverse FFT
 * Return:
 *		None (output replace in x and y)
 *===================================================================================*/
static void complex_FFT(float *x, float *y, int n, int direction)
{
  register int i, j, k;
  int N;
  float *real, *image;
  float temp;
  int le,le1,ip;
  float tx,ty,ux,uy;
  
  /* Set data array according to direction */
  if (direction == 1)
    {
      /* FFT */
      real = x;
      image = y;
    }
  else if (direction == -1)
    {
      /* Inverse FFT */
      real = y;
      image = x;
    }

  /* Determine the size of the FFT */
  N = 1;
  N = N << n;
  
  /* Bit reversal */
  j = 0;
  for (i = 0; i < N - 1; i++)
    {
      if (i < j)
        {
          /* swap real */
          temp = real[i];
          real[i] = real[j];
          real[j] = temp;
          
          /* swap image */
          temp = image[i];
          image[i] = image[j];
          image[j] = temp;
        }
      k = N / 2;
      while (k <= j)
        {
          j -= k;
          k /= 2;
        }
      j += k;
    }

  /* FFT main loop */
  for (k = 0; k < n; k++)
    {
      le1 = 1;
      le1 = le1 << k;
      le = le1 * 2;
      ux = 1.0;
      uy = 0.0;
      for (j = 0; j < le1; j++)
        {
          ux = (float)cos(j * PI / (float)le1);
          uy = - (float)sin( j * PI /(float)le1);
          for (i = j; i < N; i += le)
            {
              ip = i + le1;
              tx = real[ip]*ux - image[ip]*uy;
              ty = real[ip]*uy + image[ip]*ux;
              real[ip]  = real[i]  - tx;
              image[ip] = image[i] - ty;
              real[i]  = real[i]  + tx;
              image[i] = image[i] + ty;
            }
        }
    }
}

/*===================================================================================
 * Function: calc_pds
 *		Compute power density spectrum.
 * Arguments:
 *		float Re_fft[FRAMELEN]:			FFT real part
 *		float Im_fft[FRAMELEN]:			FFT image part
 *		double pow_spec[FRAMELEN/2]:	power density spectrum
 * Return:
 *		None (output replace in pow_spec)
 *===================================================================================*/
static void calc_pds(float Re_fft[FRAMELEN],float Im_fft[FRAMELEN],double pow_spec[FRAMELEN/2])
{
  int i;
  double pow_x;
  double max_pow,Delta;

  for (i=0;i<FRAMELEN/2;i++)
    {
      pow_x=sqrt(pow(Re_fft[i],2)+pow(Im_fft[i],2))/1024+10e-15;
      pow_spec[i]=max(20*log10(pow_x),MIN_POW);
    }

  /* Normalized to reference sound pressure level 96 dB */
  max_pow=pow_spec[0];	
  for (i=1;i<FRAMELEN/2;i++)
    {
      max_pow=max(max_pow,pow_spec[i]);
    }
  Delta=96-max_pow;
	
  for (i=0;i<FRAMELEN/2;i++)
    {
      pow_spec[i]=pow_spec[i]+Delta;
    }
}

/*===================================================================================
 * Function: find_tonal_components
 *		Detect tonal.
 * Arguments:
 *		double pow_spec[FRAMELEN/2]:	power density spectrum
 *		int tonal_flag[FRAMELEN/2]:		tonal flag
 * Return:
 *		None (output replace in tonal_flag)
 *===================================================================================*/
static void find_tonal(double pow_spec[FRAMELEN/2], int tonal_flag[FRAMELEN/2])
{
  int i,j;
  int is_tonal;	

  double tonal_spl;
  double absolute_threshold_xm;

  /* initialize tonal_flag[] */
  for(i=0;i<=511;i++)
    {
      tonal_flag[i]=0;
    }

  for (i=2;i<500;i++)
    {
      if (pow_spec[i]>pow_spec[i-1]&&pow_spec[i]>=pow_spec[i+1])
        {
          is_tonal=1;

          /* Verify it meets the condition: pow_spec[i]-pow_spec[i+j]>=7 */
          /* j={-2,2} */
          if (1<i&&i<62)
            {
              for (j=-2;j<=-2;j++)
                {
                  is_tonal=is_tonal&&pow_spec[i]-pow_spec[i+j]>=7;
                  if (is_tonal==0) break;	
                }
              if (is_tonal==1)
                {
                  for (j=2;j<=2;j++)
                    {
                      is_tonal=is_tonal&&pow_spec[i]-pow_spec[i+j]>=7;
                      if (is_tonal==0) break;	
                    }
                }

              if (is_tonal==1)
                {
                  tonal_flag[i]=1;
                }
            }

          /* j={-3:-2,2:3} */
          else if(62<=i&&i<126)
            {
              for (j=-3;j<=-2;j++)
                {
                  is_tonal=is_tonal&&pow_spec[i]-pow_spec[i+j]>=7;
                  if (is_tonal==0) break;	
                }
              if (is_tonal==1)
                {
                  for (j=2;j<=3;j++)
                    {
                      is_tonal=is_tonal&&pow_spec[i]-pow_spec[i+j]>=7;
                      if (is_tonal==0) break;	
                    }
                }

              if (is_tonal==1)
                {
                  tonal_flag[i]=1;
                }
            }

          /* j={-6:-2,2:6} */
          else if(126<=i&&i<254)
            {
              for (j=-6;j<=-2;j++)
                {
                  is_tonal=is_tonal&&pow_spec[i]-pow_spec[i+j]>=7;
                  if (is_tonal==0) break;	
                }
              if (is_tonal==1)
                {
                  for (j=2;j<=6;j++)
                    {
                      is_tonal=is_tonal&&pow_spec[i]-pow_spec[i+j]>=7;
                      if (is_tonal==0) break;	
                    }
                }
				
              if (is_tonal==1)
                {
                  tonal_flag[i]=1;
                }
            }

          /* j={-12:-2,2:12} */
          else if(254<=i&&i<500)
            {
              for (j=-12;j<=-2;j++)
                {
                  is_tonal=is_tonal&&pow_spec[i]-pow_spec[i+j]>=7;
                  if (is_tonal==0) break;	
                }
              if (is_tonal==1)
                {
                  for (j=2;j<=12;j++)
                    {
                      is_tonal=is_tonal&&pow_spec[i]-pow_spec[i+j]>=7;
                      if (is_tonal==0) break;	
                    }
                }
				
              if (is_tonal==1)
                {
                  tonal_flag[i]=1;
                }
            }
          else
            {
              is_tonal=0;
            }
        }
    }

  for(i=0;i<=511;i++)
    {
      if(tonal_flag[i]==1)
        {
          /* compute the SPL of tonal */
          tonal_spl=10*log10(pow(10,(pow_spec[i-1]/10))+pow(10,(pow_spec[i]/10))+pow(10,(pow_spec[i+1]/10)));

          if(i>=324)
            {
              absolute_threshold_xm=absolute_threshold[i]+20;
            }
          else
            {
              absolute_threshold_xm=absolute_threshold[i];
            }

          if(tonal_spl<absolute_threshold_xm)
            {
              tonal_flag[i]=0;
            }			
        }
    }
}

/*===================================================================================
 * Function: tonal_analysis
 *		Analysis the characters of tonal.
 * Arguments:
 *		float time_signal[FRAMELEN]:		input signals
 *		int framecnt_xm:					frame counter 
 *		int NTonal[100]:					buffer of the numbers of tonal
 *		int NTonal_low_frequency[100]:		buffer of the numbers of tonal
 *											in the low frequency domain
 *		float *NTonal_low_frequency_ratio:	the ratio of distribution of 
 *											the numbers of tonal
 *											in the low frequency domain
 *		float *ave_NTonal:					long-term AVE of tonal
 *		float *ave_NTonal_short:			short-term AVE of tonal
 * Return:
 *		None (output replace in NTonal_low_frequency_ratio, ave_NTonal 
 *		and ave_NTonal_short)
 *===================================================================================*/
static void tonal_analysis(float time_signal[FRAMELEN],int framecnt_xm,
                           int NTonal[100],int NTonal_low_frequency[100],
                           float *NTonal_low_frequency_ratio,float *ave_NTonal,float *ave_NTonal_short)
{
  int i;
	
  float Re_fft[FRAMELEN],Im_fft[FRAMELEN];
  int fft_size=10;
  int fft_direction=1;
	
  double pow_spec[FRAMELEN/2];

  int tonal_flag[FRAMELEN/2];

  int frame_length;
  int NTonal_total,NTonal_low_frequency_total;
	
  for (i=0;i<FRAMELEN;i++)
    {
      Re_fft[i]=(float)(time_signal[i]*hanning_window[i]);
      Im_fft[i]=0;
    }
  complex_FFT(Re_fft,Im_fft,fft_size,fft_direction);

  /* compute power density spectrum */
  calc_pds(Re_fft,Im_fft,pow_spec);

  /* detect tonal */
  find_tonal(pow_spec,tonal_flag);

  /* update NTonal, NTonal_low_frequency */
  for(i=0;i<99;i++)
    {
      NTonal[i]=NTonal[i+1];
      NTonal_low_frequency[i]=NTonal_low_frequency[i+1];
    }
  NTonal[99]=0;
  for(i=0;i<FRAMELEN/2;i++)
    {
      NTonal[99]+=tonal_flag[i];
    }
  NTonal_low_frequency[99]=0;
  for(i=0;i<INDEXOFLOWFREQUENCY;i++)
    {
      NTonal_low_frequency[99]+=tonal_flag[i];
    }

  /* compute long-term AVE and the ratio of distribution in low-frequency domain */
  if(framecnt_xm<AVE_TONAL_LENGTH)
    {
      frame_length=framecnt_xm;
    }
  else
    {
      frame_length=AVE_TONAL_LENGTH;
    }

  NTonal_total=0;
  NTonal_low_frequency_total=0;
  for(i=0;i<frame_length;i++)
    {
      NTonal_total+=NTonal[99-i];
      NTonal_low_frequency_total+=NTonal_low_frequency[99-i];
    }

  *ave_NTonal=(float)NTonal_total/frame_length;

  if(NTonal_total==0)
    {
      *NTonal_low_frequency_ratio=1;
    }
  else
    {
      *NTonal_low_frequency_ratio=(float)NTonal_low_frequency_total/NTonal_total;
    }

  /* compute the short-term AVE */
  if(framecnt_xm<AVE_TONAL_LENGTH_SHORT)
    {
      frame_length=framecnt_xm;
    }
  else
    {
      frame_length=AVE_TONAL_LENGTH_SHORT;
    }

  NTonal_total=0;
  for(i=0;i<frame_length;i++)
    {
      NTonal_total+=NTonal[99-i];
    }

  *ave_NTonal_short=(float)NTonal_total/frame_length;
}

/*===================================================================================
 * Function: spectral_tilt_analysis
 *		Spectral tilt analysis.
 * Arguments:
 *		float time_signal[FRAMELEN]:		input signals
 *		int framecnt_xm:					frame counter
 *		float spec_tilt_buf[100]:			buffer of spectral tilt
 *		float *msd_spec_tilt:				long-term MSD of spectral tilt
 *		float *msd_spec_tilt_short:			short-term MSD of spectral tilt
 *		float *frame_energy:				the energy of current frame
 * Return:
 *		None (output replace in msd_spec_tilt and msd_spec_tilt_short)
 *===================================================================================*/
static void spectral_tilt_analysis(float time_signal[FRAMELEN],int framecnt_xm,
                                   float spec_tilt_buf[100],
                                   float *msd_spec_tilt,float *msd_spec_tilt_short,float *frame_energy)
{
  int i;
  int frame_length;
	
  float r0,r1;
  float spec_tilt;
  float ave_spec_tilt;
	
  /* compute spectral tilt */
  r0 = 0;
  for (i = 0; i < FRAMELEN; i++)
    {
      r0 += time_signal[i] * time_signal[i];
    }

  r1 = 0;
  for (i = 0; i < FRAMELEN - 1; i++)
    {
      r1 += time_signal[i] * time_signal[i + 1];
    }

  if (r0 == 0)
    {
      spec_tilt = 1.0f;
    }
  else
    {
      spec_tilt = r1 / r0;
    }

  /* update spec_tilt_buf */
  for (i = 0; i < 100 - 1; i++)
    {
      spec_tilt_buf[i] = spec_tilt_buf[i + 1];
    }
  spec_tilt_buf[99] = spec_tilt;
		
  /* compute the long-term mean square deviation of the spectral tilt */
  if(framecnt_xm < SPECTRAL_TILT_LENGTH)
    {
      frame_length = framecnt_xm;
    }
  else
    {
      frame_length = SPECTRAL_TILT_LENGTH;
    }
	
  ave_spec_tilt = 0;
  for(i = 0; i < frame_length; i++)
    {
      ave_spec_tilt += spec_tilt_buf[99 - i];
    }
  ave_spec_tilt /= frame_length;

  *msd_spec_tilt = 0;
  for(i = 0; i < frame_length; i++)
    {
      *msd_spec_tilt += (spec_tilt_buf[99 - i] - ave_spec_tilt)
        * (spec_tilt_buf[99 - i] - ave_spec_tilt);
    }
  *msd_spec_tilt /= frame_length;

  /* compute the short-term mean square deviation of the spectral tilt */
  if(framecnt_xm < SPECTRAL_TILT_LENGTH_SHORT)
    {
      frame_length = framecnt_xm;
    }
  else
    {
      frame_length = SPECTRAL_TILT_LENGTH_SHORT;
    }
	
  ave_spec_tilt = 0;
  for(i = 0; i < frame_length; i++)
    {
      ave_spec_tilt += spec_tilt_buf[99 - i];
    }
  ave_spec_tilt /= frame_length;

  *msd_spec_tilt_short = 0;
  for(i = 0; i < frame_length; i++)
    {
      *msd_spec_tilt_short += (spec_tilt_buf[99 - i] - ave_spec_tilt)
        * (spec_tilt_buf[99 - i] - ave_spec_tilt);
    }
  *msd_spec_tilt_short /= frame_length;

  /* compute the energy of current frame */
  if(r0<=1)
    {
      *frame_energy=0;
    }
  else
    {
      *frame_energy=(float)(10*log(r0)/log(10));
    }
}

/*===================================================================================
 * Function: init_mode_decision
 *		Initial mode decision and boundary decisions.
 * Arguments:
 *		int framecnt:						frame counter
 *		int *framecnt_xm:					frame counter
 *		int *flag_border:					flag of current border
 *		float ave_NTonal_short:				short-term AVE of tonal
 *		float ave_NTonal:					long-term AVE of tonal
 *		float ave_NTonal_short_buf[5]:		buffer of short-term AVE of tonal
 *		float ave_NTonal_buf[5]:			buffer long-term AVE of tonal
 *		float msd_spec_tilt:				long-term MSD of spectral tilt
 *		float msd_spec_tilt_short:			short-term MSD of spectral tilt
 *		float msd_spec_tilt_buf[5]:			buffer of long-term MSD of spectral tilt
 *		float msd_spec_tilt_short_buf[5]:	buffer of short-term MSD of spectral tilt
 *		float NTonal_low_frequency_ratio:	the ratio of distribution of 
 *											the numbers of tonal
 *											in the low frequency domain
 *		float frame_energy:					the energy of current frame
 * Return:
 *		init_mode_decision_result:			the result of initial mode decision
 *		(other output replace in framecnt_xm and flag_border)
 *===================================================================================*/
static int init_mode_decision(int framecnt,int *framecnt_xm,int *flag_border,
                              float ave_NTonal_short,float ave_NTonal,
                              float ave_NTonal_short_buf[5],float ave_NTonal_buf[5],
                              float msd_spec_tilt,float msd_spec_tilt_short,
                              float msd_spec_tilt_buf[5],float msd_spec_tilt_short_buf[5],
                              float NTonal_low_frequency_ratio,float frame_energy)
{
  int i;

  int init_mode_decision_result=TBD;

  static int count_msd_st_monchhichi;
  static int count_msd_st_speech_music,count_msd_st_music_speech;
  static int flag_ave_music_speech;
  static int count_msd_st_music;
  static int border_state;
  static int count_quiet_mode;

  *flag_border=NO_BORDER;

  /*****************************************************************************/
  /* border decision according to spectral tilt */

  /* update msd_spec_tilt_buf, msd_spec_tilt_short_buf */
  for(i=0;i<5-1;i++)
    {
      msd_spec_tilt_buf[i]=msd_spec_tilt_buf[i+1];
      msd_spec_tilt_short_buf[i]=msd_spec_tilt_short_buf[i+1];
    }
  msd_spec_tilt_buf[4]=msd_spec_tilt;
  msd_spec_tilt_short_buf[4]=msd_spec_tilt_short;

  /* speech->music */
  /* find strict border of speech->music */
  if((msd_spec_tilt>=0.014)
     &&(msd_spec_tilt_short<=0.000005))
    {
      count_msd_st_monchhichi++;
    }
  else
    {
      count_msd_st_monchhichi=0;
    }
  if(((*flag_border!=BORDER_SPEECH_MUSIC_DEFINITE)
      ||(*flag_border!=BORDER_MUSIC_SPEECH_DEFINITE))
     &&(border_state!=BORDER_SPEECH_MUSIC_DEFINITE)
     &&(count_msd_st_monchhichi>=15)
     &&(*framecnt_xm>=300))
    {
      count_msd_st_monchhichi=0;
      *framecnt_xm=10;
      *flag_border=BORDER_SPEECH_MUSIC;
    }

  /* find the relative loose border of speech->music */
  if((msd_spec_tilt>=0.0025)
     &&(msd_spec_tilt_short<=0.000003))
    {
      count_msd_st_speech_music++;
    }
  else
    {
      count_msd_st_speech_music=0;
    }
  if(((*flag_border!=BORDER_SPEECH_MUSIC_DEFINITE)
      ||(*flag_border!=BORDER_MUSIC_SPEECH_DEFINITE))
     &&(border_state!=BORDER_SPEECH_MUSIC_DEFINITE)
     &&(count_msd_st_speech_music>=15)
     &&(*framecnt_xm>=300))
    {
      count_msd_st_speech_music=0;
      *framecnt_xm=10;
      *flag_border=BORDER_SPEECH_MUSIC;
    }

  /* music->speech */
  if((msd_spec_tilt_buf[0]<=0.0003)
     &&(msd_spec_tilt_short_buf[0]<=0.0002))
    {
      count_msd_st_music_speech++;
    }
  if(((*flag_border!=BORDER_SPEECH_MUSIC_DEFINITE)
      ||(*flag_border!=BORDER_MUSIC_SPEECH_DEFINITE))
     &&(border_state!=BORDER_MUSIC_SPEECH_DEFINITE)
     &&(count_msd_st_music_speech>=100)
     &&(msd_spec_tilt>=0.0008)
     &&(msd_spec_tilt_short>=0.0025)
     &&(*framecnt_xm>=20))
    {
      count_msd_st_music_speech=0;
      *framecnt_xm=10;
      *flag_border=BORDER_MUSIC_SPEECH;
    }

  /*****************************************************************************/
  /* border decision according to tonal */

  /* update ave_NTonal_short_buf, ave_NTonal_buf */
  for(i=0;i<5-1;i++)
    {
      ave_NTonal_short_buf[i]=ave_NTonal_short_buf[i+1];
      ave_NTonal_buf[i]=ave_NTonal_buf[i+1];
    }
  ave_NTonal_short_buf[4]=ave_NTonal_short;
  ave_NTonal_buf[4]=ave_NTonal;

  /* music->speech */
  if((ave_NTonal_buf[0]>=12)
     &&(ave_NTonal_buf[0]<15)
     &&(ave_NTonal_buf[0]-ave_NTonal_short_buf[0]>=5)
     &&(*framecnt_xm>=20)
     &&(ave_NTonal_short-ave_NTonal_short_buf[0]<5))
    {
      *framecnt_xm=10;
      flag_ave_music_speech=1;
      *flag_border=BORDER_MUSIC_SPEECH_DEFINITE;
    }

  /*****************************************************************************/
  /* update border decision according to energy	*/
  if(frame_energy<=60)
    {
      count_quiet_mode=0;
    }
  else
    {
      count_quiet_mode++;
    }

  if((*flag_border==BORDER_MUSIC_SPEECH)
     &&(count_quiet_mode<=5))
    {
      *flag_border=BORDER_MUSIC_SPEECH_DEFINITE;
      *framecnt_xm=10;
    }

  /*****************************************************************************/
  /* MUSIC_DEFINITE and SPEECH_DEFINITE mode decision according to short-term characters */
	
  /* ave_NTonal_short */
  if((init_mode_decision_result==TBD)
     &&(ave_NTonal_short>=19))
    {
      init_mode_decision_result=MUSIC_DEFINITE;
    }
  if((init_mode_decision_result==TBD)
     &&(ave_NTonal_short<=1.5))
    {
      init_mode_decision_result=SPEECH_DEFINITE;
    }

  /* msd_spec_tilt_short */
  if(msd_spec_tilt_short>=0.02)
    {
      init_mode_decision_result=SPEECH_DEFINITE;
    }
  if((init_mode_decision_result==TBD)
     &&(msd_spec_tilt_short<=0.00000025)
     &&(framecnt>=10))
    {
      init_mode_decision_result=MUSIC_DEFINITE;
    }

  /*****************************************************************************/
  /* SPEECH mode decision */
	
  /* flag_ave_music_speechave_NTonal_short */
  if((init_mode_decision_result==TBD)
     &&(flag_ave_music_speech==1))
    {
      if((ave_NTonal_short<=12)
         &&(*framecnt_xm<=150))
        {
          init_mode_decision_result=SPEECH;
        }
      else
        {
          flag_ave_music_speech=0;
        }
    }

  /*****************************************************************************/
  /* MUSIC_DEFINITE and SPEECH_DEFINITE mode decision */
	
  /* ave_NTonal */
  if((init_mode_decision_result==TBD)
     &&(ave_NTonal<=3))
    {
      init_mode_decision_result=SPEECH_DEFINITE;
    }
  if((init_mode_decision_result==TBD)
     &&(ave_NTonal>=15))
    {
      init_mode_decision_result=MUSIC_DEFINITE;
    }

  /* ave_NTonal_short */
  if((init_mode_decision_result==TBD)
     &&(ave_NTonal_short>=17))
    {
      init_mode_decision_result=MUSIC_DEFINITE;
    }

  /* msd_spec_tilt */
  if((init_mode_decision_result==TBD)
     &&(msd_spec_tilt>=0.01))
    {
      init_mode_decision_result=SPEECH_DEFINITE;
    }
  if((init_mode_decision_result==TBD)
     &&(framecnt>=10)
     &&(msd_spec_tilt<=0.00004))
    {
      init_mode_decision_result=MUSIC_DEFINITE;
    }

  /* NTonal_low_frequency_ratio */
  if((init_mode_decision_result==TBD)
     &&(NTonal_low_frequency_ratio<=0.91))
    {
      init_mode_decision_result=MUSIC_DEFINITE;
    }

  /*****************************************************************************/
  /* MUSIC and SPEECH mode decision */

  /* msd_spec_tilt */
  if((init_mode_decision_result==TBD)
     &&(msd_spec_tilt<=0.0002)
     &&(*framecnt_xm>=15))
    {
      init_mode_decision_result=MUSIC;
    }

  /* NTonal_low_frequency_ratio */
  if((init_mode_decision_result==TBD)
     &&(NTonal_low_frequency_ratio>=0.95))
    {
      init_mode_decision_result=SPEECH;
    }
  if((init_mode_decision_result==TBD)
     &&(NTonal_low_frequency_ratio<=0.935))
    {
      init_mode_decision_result=MUSIC;
    }

  /* the rest of the frame to SPEECH */
  if(init_mode_decision_result==TBD)
    {
      init_mode_decision_result=SPEECH;
    }

  /*****************************************************************************/
  /* MUSIC mode decision according to changes of the MSD of the spectral tilt */
	
  /* compute the changes of the MSD of the spectral tilt */
  if((msd_spec_tilt<=0.007)
     &&(init_mode_decision_result!=SPEECH_DEFINITE))
    {
      if(init_mode_decision_result!=SPEECH)
        {
          count_msd_st_music++;
        }
    }
  else
    {
      count_msd_st_music=0;
    }

  if((init_mode_decision_result!=SPEECH_DEFINITE)
     &&(count_msd_st_music>=400)
     &&(border_state!=BORDER_MUSIC_SPEECH_DEFINITE))
    {
      init_mode_decision_result=MUSIC;
    }

  /*****************************************************************************/
  /* update border flag */

  if(*flag_border!=NO_BORDER)
    {
      border_state=*flag_border;
    }

  /* update BORDER_SPEECH_MUSIC_DEFINITE */
  if(((border_state==BORDER_MUSIC_SPEECH)
      ||(border_state==BORDER_MUSIC_SPEECH_DEFINITE))
     &&(init_mode_decision_result==MUSIC_DEFINITE)
     &&(*framecnt_xm>=20))
    {
      *flag_border=BORDER_SPEECH_MUSIC_DEFINITE;
      *framecnt_xm=10;
      border_state=*flag_border;
    }

  /* update BORDER_MUSIC_SPEECH_DEFINITE */
  if(((border_state==BORDER_SPEECH_MUSIC)
      ||(border_state==BORDER_SPEECH_MUSIC_DEFINITE))
     &&(init_mode_decision_result==SPEECH_DEFINITE)
     &&(*framecnt_xm>=20))
    {
      *flag_border=BORDER_MUSIC_SPEECH_DEFINITE;
      *framecnt_xm=10;
      border_state=*flag_border;
    }

  return init_mode_decision_result;
}

/*===================================================================================
 * Function: smoothing_mode_decision
 *		Smoothing.
 * Arguments:
 *		int init_mode_decision_result:		initial mode decision
 *		int init_result_behind[100]:		buffer of past mode decisions
 *		int init_result_ahead[]:			buffer of ahead mode decisions
 *		int flag_border:					current flag of border
 *		int flag_border_buf_behind[10]:		buffer of past flags of border
 *		int flag_border_buf_ahead[]:		buffer of ahead flags of border
 *		float frame_energy:					the energy of current frame
 *		float frame_energy_buf_behind[10]:	buffer of past frame energies
 *		float frame_energy_buf_ahead[]:		buffer of ahead frame energies
 *		int smoothing_result_buf[100]:		buffer of smoothed mode decision
 * Return:
 *		mode_decision_result:				final mode decision result
 *===================================================================================*/
static int smoothing_mode_decision(int init_mode_decision_result,int init_result_behind[100],int init_result_ahead[NFRAMEAHEAD],
                                   int flag_border,int flag_border_buf_behind[10],int flag_border_buf_ahead[NFRAMEAHEAD],
                                   float frame_energy,float frame_energy_buf_behind[10],float frame_energy_buf_ahead[NFRAMEAHEAD],
                                   int smoothing_result_buf[100])
{
  int i;

  int mode_decision_result;
	
  static int num_smoothing;
  int num_music,num_speech;
  static int flag_speech_definite;
  static int count_small_energy;
  static int flag_music_definite;

  /*****************************************************************************/
  /* update data array */
	
  /* update init_result_behind, init_result_ahead */
  for(i=0;i<99;i++)
    {
      init_result_behind[i]=init_result_behind[i+1];
    }
  init_result_behind[99]=init_result_ahead[0];
	
  for(i=0;i<NFRAMEAHEAD-1;i++)
    {
      init_result_ahead[i]=init_result_ahead[i+1];
    }
  init_result_ahead[NFRAMEAHEAD-1]=init_mode_decision_result;

  /* update flag_border_buf_behind, flag_border_buf_ahead */
  for(i=0;i<9;i++)
    {
      flag_border_buf_behind[i]=flag_border_buf_behind[i+1];
    }
  flag_border_buf_behind[9]=flag_border_buf_ahead[0];

  for(i=0;i<NFRAMEAHEAD-1;i++)
    {
      flag_border_buf_ahead[i]=flag_border_buf_ahead[i+1];
    }
  flag_border_buf_ahead[NFRAMEAHEAD-1]=flag_border;

  /* update frame_energy_buf_behind, frame_energy_buf_ahead */
  for(i=0;i<9;i++)
    {
      frame_energy_buf_behind[i]=frame_energy_buf_behind[i+1];
    }
  frame_energy_buf_behind[9]=frame_energy_buf_ahead[0];

  for(i=0;i<NFRAMEAHEAD-1;i++)
    {
      frame_energy_buf_ahead[i]=frame_energy_buf_ahead[i+1];
    }
  frame_energy_buf_ahead[NFRAMEAHEAD-1]=frame_energy;

  /*****************************************************************************/
  /* smoothing according to past results */
	
  mode_decision_result=init_result_behind[99];

  /* update smoothing_result_buf */
  if(flag_border_buf_behind[9]==NO_BORDER)
    {
      for(i=0;i<99;i++)
        {
          smoothing_result_buf[i]=smoothing_result_buf[i+1];
        }
      num_smoothing++;
    }
  else
    {
      for(i=0;i<99;i++)
        {
          smoothing_result_buf[i]=TBD;
        }
      num_smoothing=1;
    }
  smoothing_result_buf[99]=init_result_behind[99];

  if(num_smoothing>=SMOOTHING_LENGTH)
    {
      num_music=0;
      num_speech=0;
		
      /* smoothed result count */
      for(i=0;i<SMOOTHING_LENGTH;i++)
        {
          if((smoothing_result_buf[100-i]==SPEECH)
             ||(smoothing_result_buf[100-i]==SPEECH_DEFINITE))
            {
              num_speech++;
            }
          else
            {
              num_music++;
            }
        }

      /* smoothing */
      if((num_speech>num_music)&&(init_mode_decision_result!=MUSIC_DEFINITE))
        {
          mode_decision_result=SPEECH;
        }
      if((num_music>num_speech)&&(init_mode_decision_result!=SPEECH_DEFINITE))
        {
          mode_decision_result=MUSIC;
        }
    }

  /*****************************************************************************/
  /* correct according to energies and ahead mode decision results */
	
  if((mode_decision_result==MUSIC)
     &&(frame_energy_buf_behind[9]<=60))
    {		
      for(i=0;i<NFRAMEAHEAD;i++)
        {
          if((init_result_ahead[i]==SPEECH_DEFINITE)
             ||(init_result_ahead[i]==SPEECH))
            {
              flag_speech_definite=1;
            }
        }
    }
  if((flag_speech_definite==1)
     &&(mode_decision_result==MUSIC))
    {
      mode_decision_result=SPEECH;
    }
  else
    {
      flag_speech_definite=0;
    }

  /*****************************************************************************/
  /* correct MUSIC mode */
	
  if(frame_energy_buf_behind[9]<=65)
    {
      count_small_energy=0;
    }
  else
    {
      count_small_energy++;
    }
  if(((flag_border_buf_ahead[NFRAMEAHEAD-1]==BORDER_SPEECH_MUSIC)
      ||(flag_border_buf_ahead[NFRAMEAHEAD-1]==BORDER_SPEECH_MUSIC_DEFINITE))
     &&(count_small_energy<=30))
    {
      flag_music_definite=1;
    }
  if((flag_music_definite==1)
     &&((mode_decision_result==SPEECH)
        ||(mode_decision_result==SPEECH_DEFINITE)))
    {
      mode_decision_result=MUSIC;
    }
  else
    {
      flag_music_definite=0;
    }

  return mode_decision_result;
}

/*===================================================================================
 * Function: classification_1024
 *		Classification every 1024-length frame, 
 *		the delay of output result according to NFRAMEAHEAD
 * Arguments:
 *		float time_signal[FRAMELEN]:	input signals
 * Return:
 *		mode_decision_result:	final classification
 *===================================================================================*/
static int classification_1024(float time_signal[FRAMELEN])
{
  int i;

  static int init_flag;
	
  static int framecnt = 0;		/* frame counter of input signals */
  static int framecnt_xm = 0;		/* frame counter for mode decision */
	
  static int NTonal[100],NTonal_low_frequency[100];			/* buffer of tonal */
  float NTonal_low_frequency_ratio;					/* the ratio of distribution of the numbers */
                                                                        /* of tonal in the low frequency domain     */
  float ave_NTonal,ave_NTonal_short;					/* the number of tonal */
  static float spec_tilt_buf[100];					/* buffer of spectral tilt */
  float msd_spec_tilt;							/* the long-term MSD of spectral tilt */
  float msd_spec_tilt_short;						/* the short-term MSD of spectral tilt */
  float frame_energy;							/* the energy of current frame */

  int init_mode_decision_result;					/* the initial mode decision */
  int flag_border=NO_BORDER;						/* flag of current border */
  static float msd_spec_tilt_buf[5],msd_spec_tilt_short_buf[5];	        /* buffer of the MSD of spectral tilt */
  static float ave_NTonal_short_buf[5],ave_NTonal_buf[5];		/* buffer of the AVE of tonal */

  int mode_decision_result;						/* final mode decision result */
  static int smoothing_result_buf[100];				        /* buffer of smoothed mode decisions */
  static int init_result_behind[100];					/* buffer of past mode decisions */
  static int init_result_ahead[NFRAMEAHEAD];			        /* buffer of ahead mode decisions */
  static int flag_border_buf_behind[10];				/* buffer of past border flags */
  static int flag_border_buf_ahead[NFRAMEAHEAD];		        /* buffer of ahead border flags */
  static float frame_energy_buf_behind[10];			        /* buffer of past energies */
  static float frame_energy_buf_ahead[NFRAMEAHEAD];	                /* buffer of ahead energies */
	
  if(init_flag==0)
    {/* initialize */
      init_flag=1;
		
      for(i=0;i<100;i++)
        {
          NTonal[i]=0;
          NTonal_low_frequency[i]=0;
        }
      for(i=0;i<5;i++)
        {
          ave_NTonal_short_buf[i]=0;
          ave_NTonal_buf[i]=0;
        }
		
      for(i=0;i<100;i++)
        {
          spec_tilt_buf[i]=0;
        }
      for(i=0;i<5;i++)
        {
          msd_spec_tilt_buf[i]=0;
          msd_spec_tilt_short_buf[i]=0;
        }
		
      for(i=0;i<10;i++)
        {
          frame_energy_buf_behind[i]=0;
        }
      for(i=0;i<NFRAMEAHEAD;i++)
        {
          frame_energy_buf_ahead[i]=0;
        }

      for(i=0;i<10;i++)
        {
          flag_border_buf_behind[i]=NO_BORDER;
        }
      for(i=0;i<NFRAMEAHEAD;i++)
        {
          flag_border_buf_ahead[i]=NO_BORDER;
        }
		
      for(i=0;i<100;i++)
        {
          init_result_behind[i]=TBD;
        }
      for(i=0;i<NFRAMEAHEAD;i++)
        {
          init_result_ahead[i]=TBD;
        }
      for(i=0;i<100;i++)
        {
          smoothing_result_buf[i]=TBD;
        }
    }
	
  framecnt++;
  framecnt_xm++;
	
  /* analysis tonal */
  tonal_analysis(time_signal,framecnt_xm,
                 NTonal,NTonal_low_frequency,
                 &NTonal_low_frequency_ratio,&ave_NTonal,&ave_NTonal_short);

  /* analysis spectral tilt */
  spectral_tilt_analysis(time_signal,framecnt_xm,
                         spec_tilt_buf,&msd_spec_tilt,&msd_spec_tilt_short,&frame_energy);

  /* initial mode decision and boundary decisions */
  init_mode_decision_result=init_mode_decision(framecnt,&framecnt_xm,&flag_border,
                                               ave_NTonal_short,ave_NTonal,ave_NTonal_short_buf,ave_NTonal_buf,
                                               msd_spec_tilt,msd_spec_tilt_short,msd_spec_tilt_buf,msd_spec_tilt_short_buf,
                                               NTonal_low_frequency_ratio,frame_energy);

  /* smoothing */
  mode_decision_result=smoothing_mode_decision(init_mode_decision_result,
                                               init_result_behind,init_result_ahead,
                                               flag_border,flag_border_buf_behind,flag_border_buf_ahead,
                                               frame_energy,frame_energy_buf_behind,frame_energy_buf_ahead,
                                               smoothing_result_buf);

  return mode_decision_result;
}

/*===================================================================================
 * Function: classification
 *		Classification.
 * Arguments:
 *		CLASSIFICATION *classfy:	struct of classification
 * Return:
 *		None (output replace in classfy->coding_mode)
 *===================================================================================*/
void classification (CLASSIFICATION *classfy)
{
  int n_frames,n_class,avg_cls,nf;

  int i;
  float time_signal[FRAMELEN];	
  int mode_decision_result;		

  n_frames = classfy->n_buffer_samples / FRAMELEN;
  for (nf = 0; nf < n_frames; nf++)
    {
      for (i = 0; i < FRAMELEN; i++)
        {
          time_signal[i] = classfy->input_samples[FRAMELEN * nf + i];	
        }

      /* classification of 1024-frame */
      mode_decision_result=classification_1024(time_signal);

      /* coding mode decision of 1024-frame */
      if((mode_decision_result==MUSIC)||(mode_decision_result==MUSIC_DEFINITE))
        {
          classfy->coding_mode = FD_MODE;
        }
      else if((mode_decision_result==SPEECH)||(mode_decision_result==SPEECH_DEFINITE))
        {
          classfy->coding_mode = TD_MODE;
        }

      classfy->class_buf[classfy->n_buf_class + nf] = classfy->coding_mode;        
      classfy->pre_mode = classfy->coding_mode;
    }

  /* merge 1024-frame results */
  classfy->n_buf_class += n_frames;
  n_class = (classfy->n_class_frames > classfy->n_buf_class)?classfy->n_buf_class:classfy->n_class_frames;    
  {
    int min_cls, max_cls;

    avg_cls = 0;
    min_cls = max_cls = classfy->class_buf[0];
    for (i = 1; i < n_class; i++)
      {
        if (classfy->class_buf[i] > max_cls)
          {
            max_cls = classfy->class_buf[i];
          }else if (classfy->class_buf[i] < min_cls)
          {
            min_cls = classfy->class_buf[i];
          }
      }
        
    avg_cls = 0;
    for (i = 0; i < n_class; i++)
      {
        if (classfy->class_buf[i] == max_cls)
          {
            avg_cls += 1;
          }
        if (classfy->class_buf[i] == min_cls)
          {
            avg_cls += -1;
          }
      }
        
    if (avg_cls > 0 )
      {
        classfy->coding_mode = max_cls;            
      }else{
      classfy->coding_mode = min_cls;            
    }
  }
    
  /* shift, save pre_mode and unused class */
  classfy->pre_mode = classfy->class_buf[n_class - 1];
  classfy->n_buf_class -= n_class;    
  for (i = 0; i < classfy->n_buf_class; i++)
    {
      classfy->class_buf[i] = classfy->class_buf[i + n_class];
    }    

  /* shift, save unused samples */
  classfy->n_buffer_samples -= FRAMELEN * n_frames;
  for (i = 0; i < classfy->n_buffer_samples; i ++)
    {
      classfy->input_samples[i] = classfy->input_samples[i + FRAMELEN * n_frames];
    }
}

/*===================================================================================
 * Function: init_classification
 *		Initialize.
 * Arguments:
 *		CLASSIFICATION *classfy:	struct of classification
 *		int  bitrate:		
 *		int  codecMode:				is switch mode
 * Return:
 *		None (output replace in classfy)
 *===================================================================================*/
void init_classification(CLASSIFICATION *classfy, int  bitrate, int codecMode)
{
  classfy->pre_mode = FD_MODE;

  classfy->n_buffer_samples = 0;
  memset(classfy->input_samples, 0, 3840 * 2 * sizeof(float));
  classfy->n_class_frames = 2;
  classfy->n_buf_class = 0;
    
  classfy->isSwitchMode = codecMode;
}
