/****************************************************************************

SC 29 Software Copyright Licencing Disclaimer:

This software module was originally developed by
  Coding Technologies

and edited by
  -

in the course of development of the ISO/IEC 13818-7 and ISO/IEC 14496-3
standards for reference purposes and its performance may not have been
optimized. This software module is an implementation of one or more tools as
specified by the ISO/IEC 13818-7 and ISO/IEC 14496-3 standards.
ISO/IEC gives users free license to this software module or modifications
thereof for use in products claiming conformance to audiovisual and
image-coding related ITU Recommendations and/or ISO/IEC International
Standards. ISO/IEC gives users the same free license to this software module or
modifications thereof for research purposes and further ISO/IEC standardisation.
Those intending to use this software module in products are advised that its
use may infringe existing patents. ISO/IEC have no liability for use of this
software module or modifications thereof. Copyright is not released for
products that do not conform to audiovisual and image-coding related ITU
Recommendations and/or ISO/IEC International Standards.
The original developer retains full right to modify and use the code for its
own purpose, assign or donate the code to a third party and to inhibit third
parties from using the code for products that do not conform to audiovisual and
image-coding related ITU Recommendations and/or ISO/IEC International Standards.
This copyright notice must be included in all copies or derivative works.
Copyright (c) ISO/IEC 2002.

 $Id: ct_polyphase.c,v 1.5.12.1 2012-04-19 09:15:33 frd Exp $

*******************************************************************************/
/*!
  \file
  \brief  Polyphase Filterbank $Revision: 1.5.12.1 $
*/

#include <math.h>
#include "ct_polyphase.h"
#include "ct_sbrconst.h"

#include <stdlib.h>

#ifndef NULL
#define NULL 0
#endif

/*!
  Linear interpolation of prototype filter

  origProt - pointer to original prototype filter
  No - length of origProt
  Lo - number of original channels
  Li - number of channels for interpolated filter
  phase - phase offset

  Written by Per Ekstrand, Coding Technologies 2007.
*/
double * interpolPrototype(const double *origProt,
                           int   No,
                           int   Lo,
                           int   Li,
                           double phase)
{
  int   i, nf;
  double n, *interProt;
  double r  = (double) Lo / Li;   /*! interpolation ratio */
  int   Ni = (int) (No/r+0.5f);   /*! length of interpolated prototype */

  if( fabs(Ni - No / r) > 1e-3 ) return NULL;

  interProt = (double *) malloc (Ni * sizeof(double));
  if(interProt == NULL) return NULL;

  for(i = 0; i < Ni; i++)
  {
    n  = i * r + phase;
    nf = (int) n;

    interProt[i] = (n-nf) * ( origProt[nf+1] - origProt[nf] ) + origProt[nf];
  }

  return interProt;
}

static double *sbrDecoderFilterbankCoefficientsInt = NULL;

#ifdef AAC_ELD


const double ldSbrDecoderFilterbankCoefficients[640] = {
    1.129580193872797e-002,  2.353059744904218e-002,  3.450718748721251e-002,  4.634695977000525e-002,
    5.918677345174197e-002,  7.325978412117062e-002,  8.829745229234007e-002,  1.042033024802571e-001,
    1.206924277410051e-001,  1.376149808913910e-001,  1.547461142258783e-001,  1.719726384566089e-001,
    1.891590407342011e-001,  2.062605107774960e-001,  2.232276864673650e-001,  2.400768261284114e-001,
    2.568176309566753e-001,  2.734977190313227e-001,  2.901491317310591e-001,  3.068186515423912e-001,
    3.235298682841570e-001,  3.403074146062977e-001,  3.571527896130669e-001,  3.740643974275026e-001,
    3.910243970160607e-001,  4.080154903861317e-001,  4.250144186334534e-001,  4.420013942269341e-001,
    4.589582896478246e-001,  4.758753745532750e-001,  4.927463828072591e-001,  5.095720854151864e-001,
    5.263554446856779e-001,  5.430990601899994e-001,  5.598052330684253e-001,  5.764734796907189e-001,
    5.930981800982896e-001,  6.096690552916387e-001,  6.261725236758639e-001,  6.425939632009995e-001,
    6.589148753746076e-001,  6.751199626157149e-001,  6.911981575264606e-001,  7.071447728928043e-001,
    7.229599104052475e-001,  7.386515025302785e-001,  7.542294504292890e-001,  7.697093346240386e-001,
    7.851012620144958e-001,  8.004165237845137e-001,  8.156523162880560e-001,  8.308039608112368e-001,
    8.458450064727010e-001,  8.607492455327098e-001,  8.754640719350776e-001,  8.899474405744183e-001,
    9.041286138017367e-001,  9.179666107725365e-001,  9.313874086278087e-001,  9.443802853939540e-001,
    9.568885413848645e-001,  9.690016637782843e-001,  9.807691702375303e-001,  9.927543720639498e-001,
    1.001463112557766e+000,  1.006893331637123e+000,  1.012508393574432e+000,  1.017729040219375e+000,
    1.022470190536100e+000,  1.026615653698808e+000,  1.030198648769593e+000,  1.033205850580933e+000,
    1.035694432087486e+000,  1.037683165297586e+000,  1.039227995800217e+000,  1.040349586463588e+000,
    1.041086497214721e+000,  1.041443375950143e+000,  1.041434355650865e+000,  1.041043184216171e+000,
    1.040262316588456e+000,  1.039061496136853e+000,  1.037422300157921e+000,  1.035311720204252e+000,
    1.032712952177121e+000,  1.029600494883906e+000,  1.025966756910904e+000,  1.021798805583990e+000,
    1.017100128250049e+000,  1.011867706519706e+000,  1.006109248754940e+000,  9.998285752401580e-001,
    9.930379854679836e-001,  9.857387823493258e-001,  9.779405164766706e-001,  9.696426101291272e-001,
    9.608519516143015e-001,  9.515674613550604e-001,  9.417975696327747e-001,  9.315442093447622e-001,
    9.208194746232827e-001,  9.096310803629866e-001,  8.979959173503500e-001,  8.859232320517536e-001,
    8.734366852542127e-001,  8.605542791988831e-001,  8.472987145504696e-001,  8.336863467961255e-001,
    8.197387292306723e-001,  8.054701312929008e-001,  7.908995350037713e-001,  7.760385598209244e-001,
    7.609051036128973e-001,  7.455111681431031e-001,  7.298745530879272e-001,  7.140087729493950e-001,
    6.979336851549095e-001,  6.816667882498023e-001,  6.652304141388827e-001,  6.486437667370537e-001,
    6.319284031798550e-001,  6.151031151692835e-001,  5.981877665956570e-001,  5.811992722116214e-001,
    5.641522833259215e-001,  5.470652177576862e-001,  5.299509559653194e-001,  5.128557121424191e-001,
   -4.956175421414453e-001, -4.782650346610896e-001, -4.609828932783459e-001, -4.437530233023859e-001,
   -4.265950246465440e-001, -4.095160467543179e-001, -3.925409172155113e-001, -3.756821671788237e-001,
   -3.589626517817934e-001, -3.423942311297658e-001, -3.259993851088293e-001, -3.097861805973821e-001,
   -2.937724988593393e-001, -2.779637821990255e-001, -2.623749159488041e-001, -2.470098299603623e-001,
   -2.318815478758375e-001, -2.169925682529340e-001, -2.023548005388463e-001, -1.879711746686855e-001,
   -1.738542127021508e-001, -1.600061812296078e-001, -1.464389150679625e-001, -1.331544923127771e-001,
   -1.201628679722633e-001, -1.074630704470568e-001, -9.506966959632511e-002, -8.298103104739203e-002,
   -7.120356992726613e-002, -5.973741829536090e-002, -4.859005767016811e-002, -3.775928110298274e-002,
   -2.726484300186575e-002, -1.711323992709580e-002, -7.298197371320593e-003,  2.184256929356781e-003,
    1.132324047372148e-002,  2.012236990754980e-002,  2.857528272530154e-002,  3.666942822678171e-002,
    4.439683978044157e-002,  5.177964768870787e-002,  5.881296711410786e-002,  6.550209046893848e-002,
    7.184073822817207e-002,  7.783299328224960e-002,  8.347150698567406e-002,  8.875756217893037e-002,
    9.368651761350569e-002,  9.826251129465624e-002,  1.024804711677230e-001,  1.063454554357498e-001,
    1.098551252869576e-001,  1.130180022553412e-001,  1.158358935177899e-001,  1.183233335449968e-001,
    1.204854506722672e-001,  1.223371395264402e-001,  1.238868653862843e-001,  1.251477258491527e-001,
    1.261262023246478e-001,  1.268280540744526e-001,  1.272498700590511e-001,  1.273590703506806e-001,
    1.274567595465545e-001,  1.275561350483646e-001,  1.273648326872248e-001,  1.269415772180714e-001,
    1.262995646340671e-001,  1.254605188749804e-001,  1.244269583009826e-001,  1.232131583108813e-001,
    1.218183974842866e-001,  1.202545652840080e-001,  1.185243106889108e-001,  1.166399102636992e-001,
    1.146042249339280e-001,  1.124296184976912e-001,  1.101215600923314e-001,  1.076972053405737e-001,
    1.051641975499523e-001,  1.025397604985405e-001,  9.982957934346254e-002,  9.705239536075722e-002,
    9.421624116597689e-002,  9.133590931873967e-002,  8.841813387276727e-002,  8.547715661443602e-002,
    8.251962055343706e-002,  7.955570759229536e-002,  7.657649751612349e-002,  7.360559211914287e-002,
    7.064948295960993e-002,  6.771675107480543e-002,  6.480448458935215e-002,  6.192692754258131e-002,
    5.911363249658311e-002,  5.637219228757212e-002,  5.368313072045600e-002,  5.105620793438655e-002,
    4.849284995895640e-002,  4.599068181839981e-002,  4.355568588898841e-002,  4.125570251909672e-002,
    3.907137550527191e-002,  3.696342556744636e-002,  3.493300140502248e-002,  3.298151059524886e-002,
    3.110861245410919e-002,  2.931525594774175e-002,  2.760090729801069e-002,  2.597956638848436e-002,
    2.443433592149451e-002,  2.296470793543091e-002,  2.156304510969632e-002,  2.023524610221679e-002,
    1.897505817503749e-002,  1.778248750467421e-002,  1.665187994388476e-002,  1.557759513377242e-002,
    1.456208586604537e-002,  1.361072086117313e-002,  1.270747042064656e-002,  1.186210743261470e-002,
    1.106958962776399e-002,  1.033126278863177e-002,  9.640298325700842e-003,  8.996371481700806e-003,
   -8.407748878436545e-003, -7.876393114319395e-003, -7.380543918629573e-003, -6.925141135202262e-003,
   -6.500502521462604e-003, -6.109178606718115e-003, -5.741103163221257e-003, -5.394569608919965e-003,
   -5.063851046064050e-003, -4.754191853611012e-003, -4.448993249380505e-003, -4.133639756278191e-003,
   -3.811612348723333e-003, -3.505531318950422e-003, -3.209092846617964e-003, -2.927159436740159e-003,
   -2.653818578698405e-003, -2.396404013961463e-003, -2.152379960589273e-003, -1.924844672908215e-003,
   -1.699160580023900e-003, -1.480542563288228e-003, -1.283280633901446e-003, -1.131859661378862e-003,
   -9.730460256556873e-004, -7.677634115875747e-004, -5.599347984905645e-004, -3.337966579125254e-004,
   -9.099722643476421e-005,  1.498231621816041e-004,  4.366447012116811e-004,  6.307841647560053e-004,
    6.150316826138937e-004,  8.990255827053560e-004,  1.232134364570107e-003,  1.471167206249042e-003,
    1.697652664777771e-003,  1.985825255428654e-003,  2.172866052963961e-003,  1.812176023993582e-003,
    1.344657262814793e-003,  9.373975348172919e-004,  5.621720998949145e-004,  2.048498552413189e-004,
   -2.004822830002534e-004, -6.169854804735951e-004, -1.061498982103114e-003, -1.594860949611097e-003,
   -2.124647831574725e-003, -2.621537051750861e-003, -3.064311083207632e-003, -3.460362845825662e-003,
   -3.794425324215804e-003, -4.091032597247918e-003, -4.369553676668050e-003, -4.554811297024067e-003,
   -4.663276675479689e-003, -4.722567636185647e-003, -4.704321497976561e-003, -4.636227793039124e-003,
   -4.517190210387324e-003, -4.351667566540186e-003, -4.135130493071822e-003, -3.870851645947402e-003,
   -3.597475533950260e-003, -3.318857985461042e-003, -3.000422543655664e-003, -2.658042081080524e-003,
   -2.292813563887493e-003, -1.914114740669928e-003, -1.525818616748839e-003, -1.156680209049319e-003,
   -7.804546272743493e-004, -4.268574601396473e-004, -1.324291707264515e-004,  1.218226450050751e-004,
    3.189336138130849e-004,  4.749931197951235e-004,  5.970696819774243e-004,  6.673250213055329e-004,
    6.887783835812338e-004,  6.766320515830324e-004,  6.944123176012471e-004,  7.139919634325070e-004,
    7.154123487609100e-004,  7.376101027486600e-004,  6.976561203768226e-004,  5.721223454434728e-004,
    2.934875643581191e-004,  1.092526149391273e-004,  6.415402443848103e-004,  1.194730618383423e-003,
    1.557112059887280e-003,  1.891971801393744e-003,  2.225524159129023e-003,  2.530906981099261e-003,
    2.719749515067397e-003,  2.729136737522100e-003,  2.703019498899013e-003,  2.630471852319136e-003,
    2.470456304276468e-003,  2.239142906871446e-003,  2.033465291493264e-003,  1.948069005335563e-003,
    1.725029670030533e-003,  1.417366709895927e-003,  1.127141815310061e-003,  8.089811988213151e-004,
    4.708009521678285e-004,  7.882620739833088e-005, -2.998739993995956e-004, -4.733148292475610e-004,
   -5.791145447913150e-004, -6.754935404082003e-004, -8.029620210721900e-004, -9.726698841994444e-004,
   -1.196637962311630e-003, -1.292865844760059e-003, -1.146268465739874e-003, -1.040598055074471e-003,
   -9.767709065548874e-004, -9.294665200453614e-004, -9.862027119530482e-004, -1.047654674829846e-003,
   -1.099000599887377e-003, -1.151795860160292e-003, -1.194743370333155e-003, -1.250742797799558e-003,
    1.287819050086379e-003,  1.263569296641556e-003,  1.226113111394085e-003,  1.177515087338257e-003,
    1.122503050159859e-003,  1.089428846944533e-003,  1.054963366189962e-003,  9.019128558297515e-004,
    7.847839620863715e-004,  6.205675927856794e-004,  3.157663628445906e-004,  2.556449844935384e-004,
    2.520606580606257e-004,  2.346980949474655e-004,  2.060394037017961e-004,  1.635905995590986e-004,
    1.176237128375623e-004,  6.193369904730005e-005,  3.568554800150508e-005,  2.443161189273522e-005,
    1.334090914042349e-005,  2.853437194757816e-006, -1.039263591111469e-004,  5.144969377044875e-005,
    9.711681816385056e-005,  2.472023910553232e-005,  5.397064424090302e-005,  6.487880719449901e-005,
   -5.192444140699947e-005, -9.204876089551197e-005, -1.815837353167847e-004, -3.595054179561440e-004,
   -5.901617707607606e-007,  1.831121301698088e-004,  9.755685190624611e-005,  6.606461762989423e-005,
    3.799971890923797e-005,  4.150075391929448e-005,  5.021905476506264e-005,  5.861800137434713e-005,
    2.126364641291926e-005,  1.181077582797280e-004,  9.990757789944374e-005,  1.035782617124906e-004,
    8.870181845310037e-005,  5.533953373249822e-005,  1.580188994455254e-005,  1.277184430250593e-006,
    5.009913312943629e-006,  1.499170392246774e-005,  2.241545750231630e-005,  3.628511258723260e-005,
    2.406516798531014e-005,  2.515118233957011e-005,  3.759629789955498e-005,  5.408154543124121e-005,
    4.493916063285122e-005,  2.806963579578946e-005,  2.364518513682831e-005,  1.260639764582286e-005,
   -2.599467772603631e-008, -1.774108392496017e-005, -5.889276659458115e-006, -4.663777919108619e-005,
   -2.078886359425321e-004, -2.131405580107761e-004, -1.784192600231068e-004, -1.744841754193053e-004,
   -1.728672507238372e-004, -1.885286127508226e-004, -2.078299015661617e-004, -2.123671573189573e-004,
   -2.415166002501312e-004, -2.217025456251449e-004, -9.907630821710970e-005, -8.039231481768845e-005,
   -7.934509417722400e-005, -5.874199358780108e-005, -5.449816072329412e-005, -4.489491034408147e-005,
   -3.498285982359981e-005, -1.748284921486958e-005, -9.075430772832575e-006, -1.052707430241351e-005,
   -6.538878366985722e-006,  2.206341308073472e-005,  1.769261935287328e-004,  6.418658561385058e-005,
   -8.882305312548962e-005, -1.721347222211949e-005, -6.093372716385583e-005, -7.679955330373515e-005,
    7.194151087015007e-005,  7.245095937243279e-005,  7.870354371072524e-005,  5.822201682995846e-004,
    2.666444630171025e-004,  7.872592352725688e-005,  7.095886893185526e-005,  5.643103068471008e-005,
    6.904415362098980e-005,  4.694251739991356e-005,  3.367998338617662e-005,  6.481921021601837e-005,
    6.582328030188790e-005, -4.256442530773449e-005,  4.939392400898679e-005,  5.272982009116034e-005,
    4.005269212731273e-005,  2.461876679726978e-005,  4.469729032194765e-006,  3.798519731621893e-007,
    1.374896222030490e-006,  3.965363805500215e-006,  7.300588863934780e-006,  1.168894474770061e-005,
    8.563819899447630e-006,  8.975977837330335e-006,  2.800455533708622e-005,  2.015445311139832e-005,
    1.125134651175812e-005,  5.869707265615299e-006,  1.013259758329981e-005,  1.088325131492173e-005,
    7.167101260771279e-006,  4.840577540089826e-006, -1.469933448634890e-005, -8.010079089953001e-006,
   -3.299004046633323e-005, -4.373302115187172e-005, -3.177468256997963e-005, -2.976824036182567e-005,
   -2.464228015326852e-005, -1.606050838620834e-005, -6.261944255489322e-006,  4.591009581217994e-007,
    1.395220723090848e-005,  1.622786214398703e-005, -2.043464113212971e-006, -1.653463907257247e-006,
   -1.551250801467300e-008, -1.907927361317977e-006, -9.607068622268791e-007, -4.636105364510011e-007,
   -2.765649762593200e-007, -1.922074581855119e-006, -9.897194091136331e-007, -7.873304717454037e-008,
    2.945239208477290e-008, -2.757610624807679e-006, -1.402925247695813e-005, -9.388962780643742e-006,
    2.068297421740023e-005,  1.496435902895210e-007,  6.757014945674924e-009, -2.778618354859861e-007,
   -1.569003268449803e-006, -1.089500601234349e-006, -9.870547653835426e-007,  3.867483283567218e-005,
   -1.232693496472088e-005,  9.464782951082177e-007,  8.254429452094225e-007,  4.883304950437536e-007,
   -2.066961713890010e-007,  5.158212471036245e-009,  2.267731106642486e-007, -4.880844550713951e-008,
    3.361682183852576e-006,  4.677015459111491e-006,  2.820292122791583e-008,  5.143614846654519e-007,
    3.818588614859347e-009,  1.737276553950212e-007,  1.876022048145804e-007, -2.986488593070417e-009,
   -1.409927495646886e-008, -6.977078748707401e-008, -1.280675520205100e-008, -2.222072007942510e-009,
   -1.775191290895584e-009, -1.686136654621906e-009,  5.818594642226675e-006,  2.150883991167946e-006,
    2.714879009950152e-007, -2.567964804401197e-008,  2.041128570435378e-006,  3.262753594084781e-006,
    3.567581483749161e-006,  4.083718802566134e-006,  5.364807253588177e-006,  4.178050149840223e-006,
    5.189086332701670e-006,  3.357218747491756e-006,  6.310207878018869e-006,  5.924001540927652e-006,
    5.161606640348293e-006,  3.377814811745950e-006,  1.323267689777069e-006, -1.074716688428712e-007,
   -3.561585382456484e-006, -4.518603099564185e-006,  7.301956971603966e-007,  5.891904775161025e-007,
    2.801882088134371e-008,  6.322770332405526e-007,  2.542598385847351e-007,  1.272704908592385e-007,
    8.226599990523664e-008,  5.433718768789140e-007,  4.211177232106135e-007,  3.552991527555180e-008,
   -1.398913109540774e-008,  1.356727552196146e-006, -1.706941020342299e-005,  1.013575160981381e-005,
   -2.285562946018590e-005, -8.908041185396514e-008, -9.597515277415496e-009, -3.225913527455964e-007,
    1.070242712585309e-006,  6.293002327021578e-007,  3.575650976036433e-007,  2.722295965060517e-005,
    8.676848186676888e-006,  3.428660858940255e-007,  4.767793949944890e-007,  3.330981930777764e-007,
    2.399696144635756e-007,  7.326611439066549e-009,  1.349943693297681e-007, -5.393555749348494e-008,
    3.629067065524143e-006, -5.690530948134642e-006,  1.387566465624550e-008,  2.443085172403935e-007,
    1.723217058490933e-009,  7.391973323448250e-008,  5.303527922331415e-008, -8.883499047404846e-010,
   -3.870536804891648e-009, -1.846547564287500e-008, -4.244090917065736e-009, -4.013524925634108e-009,
   -6.325664562585882e-010, -6.025110605409611e-010,  1.620171502086309e-006,  5.490569954646963e-007,
    6.355303179925355e-008, -5.426597100684762e-009,  4.292861814894369e-007,  6.834209542421138e-007,
    7.099633014995863e-007,  8.109951846981774e-007,  4.118359768898598e-007,  6.571760029213382e-007,
};

#endif /* #ifdef AAC_ELD */

const double stdSbrDecoderFilterbankCoefficients[640] = {
                  0,  -0.00055252865047,  -0.00056176925738,  -0.00049475180896,
  -0.00048752279712,  -0.00048937912498,  -0.00050407143497,  -0.00052265642972,
  -0.00054665656337,  -0.00056778025613,  -0.00058709304852,  -0.00061327473938,
  -0.00063124935319,  -0.00065403333621,  -0.00067776907764,  -0.00069416146273,
  -0.00071577364744,  -0.00072550431222,  -0.00074409418541,  -0.00074905980532,
  -0.00076813719270,  -0.00077248485949,  -0.00078343322877,  -0.00077798694927,
  -0.00078036647100,  -0.00078014496257,  -0.00077579773310,  -0.00076307935757,
  -0.00075300014201,  -0.00073193571525,  -0.00072153919876,  -0.00069179375372,
  -0.00066504150893,  -0.00063415949025,  -0.00059461189330,  -0.00055645763906,
  -0.00051455722108,  -0.00046063254803,  -0.00040951214522,  -0.00035011758756,
  -0.00028969811748,  -0.00020983373440,  -0.00014463809349,  -0.00006173344072,
   0.00001349497418,   0.00010943831274,   0.00020430170688,   0.00029495311041,
   0.00040265402160,   0.00051073884952,   0.00062393761391,   0.00074580258865,
   0.00086084433262,   0.00098859883015,   0.00112501551307,   0.00125778846475,
   0.00139024948272,   0.00154432198471,   0.00168680832531,   0.00183482654224,
   0.00198411407369,   0.00214615835557,   0.00230172547746,   0.00246256169126,
   0.00262017586902,   0.00278704643465,   0.00294694477165,   0.00311254206525,
   0.00327396134847,   0.00344188741828,   0.00360082681231,   0.00376039229104,
   0.00392074323703,   0.00408197531935,   0.00422642692270,   0.00437307196781,
   0.00452098527825,   0.00466064606118,   0.00479325608498,   0.00491376035745,
   0.00503930226013,   0.00514073539032,   0.00524611661324,   0.00534716811982,
   0.00541967759307,   0.00548760401507,   0.00554757145088,   0.00559380230045,
   0.00562206432097,   0.00564551969164,   0.00563891995151,   0.00562661141932,
   0.00559171286630,   0.00554043639400,   0.00547537830770,   0.00538389758970,
   0.00527157587272,   0.00513822754514,   0.00498396877629,   0.00481094690600,
   0.00460395301471,   0.00438018617447,   0.00412516423270,   0.00384564081246,
   0.00354012465507,   0.00320918858098,   0.00284467578623,   0.00245085400321,
   0.00202741761850,   0.00157846825768,   0.00109023290512,   0.00058322642480,
   0.00002760451905,  -0.00054642808664,  -0.00115681355227,  -0.00180394725893,
  -0.00248267236449,  -0.00319337783900,  -0.00394011240522,  -0.00472225962400,
  -0.00553372111088,  -0.00637922932685,  -0.00726158168517,  -0.00817982333726,
  -0.00913253296085,  -0.01011502154986,  -0.01113155480321,  -0.01218499959508,
   0.01327182200351,   0.01439046660792,   0.01554055533423,   0.01673247129989,
   0.01794333813443,   0.01918724313698,   0.02045317933555,   0.02174675502535,
   0.02306801692862,   0.02441609920285,   0.02578758475467,   0.02718594296329,
   0.02860721736385,   0.03005026574279,   0.03150176087389,   0.03297540810337,
   0.03446209487686,   0.03596975605542,   0.03748128504252,   0.03900536794745,
   0.04053491705584,   0.04206490946367,   0.04360975421304,   0.04514884056413,
   0.04668430272642,   0.04821657200672,   0.04973857556014,   0.05125561555216,
   0.05276307465207,   0.05424527683589,   0.05571736482138,   0.05716164501299,
   0.05859156836260,   0.05998374801761,   0.06134551717207,   0.06268578081172,
   0.06397158980681,   0.06522471064380,   0.06643675122104,   0.06760759851228,
   0.06870438283512,   0.06976302447127,   0.07076287107266,   0.07170026731102,
   0.07256825833083,   0.07336202550803,   0.07410036424342,   0.07474525581194,
   0.07531373362019,   0.07580083586584,   0.07619924793396,   0.07649921704119,
   0.07670934904245,   0.07681739756964,   0.07682300113923,   0.07672049241746,
   0.07650507183194,   0.07617483218536,   0.07573057565061,   0.07515762552870,
   0.07446643947564,   0.07364060057620,   0.07267746427299,   0.07158263647903,
   0.07035330735093,   0.06896640131951,   0.06745250215166,   0.06576906686508,
   0.06394448059633,   0.06196027790387,   0.05981665708090,   0.05751526919867,
   0.05504600343009,   0.05240938217366,   0.04959786763445,   0.04663033051701,
   0.04347687821958,   0.04014582784127,   0.03664181168133,   0.03295839306691,
   0.02908240060125,   0.02503075618909,   0.02079970728622,   0.01637012582228,
   0.01176238327857,   0.00696368621617,   0.00197656014503,  -0.00320868968304,
  -0.00857117491366,  -0.01412888273558,  -0.01988341292573,  -0.02582272888064,
  -0.03195312745332,  -0.03827765720822,  -0.04478068215856,  -0.05148041767934,
  -0.05837053268336,  -0.06544098531359,  -0.07269433008129,  -0.08013729344279,
  -0.08775475365593,  -0.09555333528914,  -0.10353295311463,  -0.11168269317730,
  -0.12000779846800,  -0.12850028503878,  -0.13715517611934,  -0.14597664911870,
  -0.15496070710605,  -0.16409588556669,  -0.17338081721706,  -0.18281725485142,
  -0.19239667457267,  -0.20212501768103,  -0.21197358538056,  -0.22196526964149,
  -0.23206908706791,  -0.24230168845974,  -0.25264803095722,  -0.26310532994603,
  -0.27366340405625,  -0.28432141891085,  -0.29507167170646,  -0.30590985751916,
  -0.31682789136456,  -0.32781137272105,  -0.33887226938665,  -0.34999141229310,
   0.36115899031355,   0.37237955463061,   0.38363500139043,   0.39492117615675,
   0.40623176767625,   0.41756968968409,   0.42891199207373,   0.44025537543665,
   0.45159965356824,   0.46293080852757,   0.47424532146115,   0.48552530911099,
   0.49677082545707,   0.50798175000434,   0.51912349702391,   0.53022408956855,
   0.54125534487322,   0.55220512585061,   0.56307891401370,   0.57385241316923,
   0.58454032354679,   0.59511230862496,   0.60557835389180,   0.61591099320291,
   0.62612426956055,   0.63619801077286,   0.64612696959461,   0.65590163024671,
   0.66551398801627,   0.67496631901712,   0.68423532934598,   0.69332823767032,
   0.70223887193539,   0.71094104263095,   0.71944626349561,   0.72774489002994,
   0.73582117582769,   0.74368278636488,   0.75131374561237,   0.75870807608242,
   0.76586748650939,   0.77277808813327,   0.77942875190216,   0.78583531203920,
   0.79197358416424,   0.79784664137700,   0.80344857518505,   0.80876950044491,
   0.81381912706217,   0.81857760046468,   0.82304198905409,   0.82722753473360,
   0.83110384571520,   0.83469373618402,   0.83797173378865,   0.84095413924722,
   0.84362382812005,   0.84598184698206,   0.84803157770763,   0.84978051984268,
   0.85119715249343,   0.85230470352147,   0.85310209497017,   0.85357205739107,
   0.85373856005937,   0.85357205739107,   0.85310209497017,   0.85230470352147,
   0.85119715249343,   0.84978051984268,   0.84803157770763,   0.84598184698206,
   0.84362382812005,   0.84095413924722,   0.83797173378865,   0.83469373618402,
   0.83110384571520,   0.82722753473360,   0.82304198905409,   0.81857760046468,
   0.81381912706217,   0.80876950044491,   0.80344857518505,   0.79784664137700,
   0.79197358416424,   0.78583531203920,   0.77942875190216,   0.77277808813327,
   0.76586748650939,   0.75870807608242,   0.75131374561237,   0.74368278636488,
   0.73582117582769,   0.72774489002994,   0.71944626349561,   0.71094104263095,
   0.70223887193539,   0.69332823767032,   0.68423532934598,   0.67496631901712,
   0.66551398801627,   0.65590163024671,   0.64612696959461,   0.63619801077286,
   0.62612426956055,   0.61591099320291,   0.60557835389180,   0.59511230862496,
   0.58454032354679,   0.57385241316923,   0.56307891401370,   0.55220512585061,
   0.54125534487322,   0.53022408956855,   0.51912349702391,   0.50798175000434,
   0.49677082545707,   0.48552530911099,   0.47424532146115,   0.46293080852757,
   0.45159965356824,   0.44025537543665,   0.42891199207373,   0.41756968968409,
   0.40623176767625,   0.39492117615675,   0.38363500139043,   0.37237955463061,
  -0.36115899031355,  -0.34999141229310,  -0.33887226938665,  -0.32781137272105,
  -0.31682789136456,  -0.30590985751916,  -0.29507167170646,  -0.28432141891085,
  -0.27366340405625,  -0.26310532994603,  -0.25264803095722,  -0.24230168845974,
  -0.23206908706791,  -0.22196526964149,  -0.21197358538056,  -0.20212501768103,
  -0.19239667457267,  -0.18281725485142,  -0.17338081721706,  -0.16409588556669,
  -0.15496070710605,  -0.14597664911870,  -0.13715517611934,  -0.12850028503878,
  -0.12000779846800,  -0.11168269317730,  -0.10353295311463,  -0.09555333528914,
  -0.08775475365593,  -0.08013729344279,  -0.07269433008129,  -0.06544098531359,
  -0.05837053268336,  -0.05148041767934,  -0.04478068215856,  -0.03827765720822,
  -0.03195312745332,  -0.02582272888064,  -0.01988341292573,  -0.01412888273558,
  -0.00857117491366,  -0.00320868968304,   0.00197656014503,   0.00696368621617,
   0.01176238327857,   0.01637012582228,   0.02079970728622,   0.02503075618909,
   0.02908240060125,   0.03295839306691,   0.03664181168133,   0.04014582784127,
   0.04347687821958,   0.04663033051701,   0.04959786763445,   0.05240938217366,
   0.05504600343009,   0.05751526919867,   0.05981665708090,   0.06196027790387,
   0.06394448059633,   0.06576906686508,   0.06745250215166,   0.06896640131951,
   0.07035330735093,   0.07158263647903,   0.07267746427299,   0.07364060057620,
   0.07446643947564,   0.07515762552870,   0.07573057565061,   0.07617483218536,
   0.07650507183194,   0.07672049241746,   0.07682300113923,   0.07681739756964,
   0.07670934904245,   0.07649921704119,   0.07619924793396,   0.07580083586584,
   0.07531373362019,   0.07474525581194,   0.07410036424342,   0.07336202550803,
   0.07256825833083,   0.07170026731102,   0.07076287107266,   0.06976302447127,
   0.06870438283512,   0.06760759851228,   0.06643675122104,   0.06522471064380,
   0.06397158980681,   0.06268578081172,   0.06134551717207,   0.05998374801761,
   0.05859156836260,   0.05716164501299,   0.05571736482138,   0.05424527683589,
   0.05276307465207,   0.05125561555216,   0.04973857556014,   0.04821657200672,
   0.04668430272642,   0.04514884056413,   0.04360975421304,   0.04206490946367,
   0.04053491705584,   0.03900536794745,   0.03748128504252,   0.03596975605542,
   0.03446209487686,   0.03297540810337,   0.03150176087389,   0.03005026574279,
   0.02860721736385,   0.02718594296329,   0.02578758475467,   0.02441609920285,
   0.02306801692862,   0.02174675502535,   0.02045317933555,   0.01918724313698,
   0.01794333813443,   0.01673247129989,   0.01554055533423,   0.01439046660792,
  -0.01327182200351,  -0.01218499959508,  -0.01113155480321,  -0.01011502154986,
  -0.00913253296085,  -0.00817982333726,  -0.00726158168517,  -0.00637922932685,
  -0.00553372111088,  -0.00472225962400,  -0.00394011240522,  -0.00319337783900,
  -0.00248267236449,  -0.00180394725893,  -0.00115681355227,  -0.00054642808664,
   0.00002760451905,   0.00058322642480,   0.00109023290512,   0.00157846825768,
   0.00202741761850,   0.00245085400321,   0.00284467578623,   0.00320918858098,
   0.00354012465507,   0.00384564081246,   0.00412516423270,   0.00438018617447,
   0.00460395301471,   0.00481094690600,   0.00498396877629,   0.00513822754514,
   0.00527157587272,   0.00538389758970,   0.00547537830770,   0.00554043639400,
   0.00559171286630,   0.00562661141932,   0.00563891995151,   0.00564551969164,
   0.00562206432097,   0.00559380230045,   0.00554757145088,   0.00548760401507,
   0.00541967759307,   0.00534716811982,   0.00524611661324,   0.00514073539032,
   0.00503930226013,   0.00491376035745,   0.00479325608498,   0.00466064606118,
   0.00452098527825,   0.00437307196781,   0.00422642692270,   0.00408197531935,
   0.00392074323703,   0.00376039229104,   0.00360082681231,   0.00344188741828,
   0.00327396134847,   0.00311254206525,   0.00294694477165,   0.00278704643465,
   0.00262017586902,   0.00246256169126,   0.00230172547746,   0.00214615835557,
   0.00198411407369,   0.00183482654224,   0.00168680832531,   0.00154432198471,
   0.00139024948272,   0.00125778846475,   0.00112501551307,   0.00098859883015,
   0.00086084433262,   0.00074580258865,   0.00062393761391,   0.00051073884952,
   0.00040265402160,   0.00029495311041,   0.00020430170688,   0.00010943831274,
   0.00001349497418,  -0.00006173344072,  -0.00014463809349,  -0.00020983373440,
  -0.00028969811748,  -0.00035011758756,  -0.00040951214522,  -0.00046063254803,
  -0.00051455722108,  -0.00055645763906,  -0.00059461189330,  -0.00063415949025,
  -0.00066504150893,  -0.00069179375372,  -0.00072153919876,  -0.00073193571525,
  -0.00075300014201,  -0.00076307935757,  -0.00077579773310,  -0.00078014496257,
  -0.00078036647100,  -0.00077798694927,  -0.00078343322877,  -0.00077248485949,
  -0.00076813719270,  -0.00074905980532,  -0.00074409418541,  -0.00072550431222,
  -0.00071577364744,  -0.00069416146273,  -0.00067776907764,  -0.00065403333621,
  -0.00063124935319,  -0.00061327473938,  -0.00058709304852,  -0.00056778025613,
  -0.00054665656337,  -0.00052265642972,  -0.00050407143497,  -0.00048937912498,
  -0.00048752279712,  -0.00049475180896,  -0.00056176925738,  -0.00055252865047
};

static float  X[MAX_NUM_ELEMENTS][MAX_NUM_CHANNELS][320];
static float  X2[MAX_NUM_ELEMENTS][MAX_NUM_CHANNELS][2*320];
static float  X3[MAX_NUM_ELEMENTS][MAX_NUM_CHANNELS][160];

static double V[MAX_NUM_ELEMENTS][MAX_NUM_CHANNELS][1280];

static const double * sbrDecoderFilterbankCoefficientsAna = stdSbrDecoderFilterbankCoefficients;
static const double * sbrDecoderFilterbankCoefficientsSyn = stdSbrDecoderFilterbankCoefficients;

static struct M {
  double real[32][64];
  double imag[32][64];
} M;

static struct M2 M2; /* defined in ct_polyphase.h */

static struct N N;   /* defined in ct_polyphase.h */

void
#ifdef AAC_ELD
InitSbrAnaFilterbank(int ldsbr, int numBands)
#else
InitSbrAnaFilterbank(int numBands)
#endif
{
  int k,l;
  double twiddleFac = 0.5 * (((double)numBands)/32.0);
  double gain = 2.0;

#ifdef AAC_ELD

  if (ldsbr) {

    if(sbrDecoderFilterbankCoefficientsInt != 0) free(sbrDecoderFilterbankCoefficientsInt);
    sbrDecoderFilterbankCoefficientsAna = ldSbrDecoderFilterbankCoefficients;
    sbrDecoderFilterbankCoefficientsInt = interpolPrototype(sbrDecoderFilterbankCoefficientsAna,
                                                          640, 64, 32, 0.5);

  for ( k=0; k<32; k++ ) {
    for ( l=0; l<64; l++ ) {
      M.real[k][l] = gain * cos((PI/64)*(k+0.5)*(2*l-95));
#ifndef LOW_POWER_SBR
      M.imag[k][l] = gain * sin((PI/64)*(k+0.5)*(2*l-95));
#endif
    }
  }
  } else {
#endif /* AAC_ELD */
    if(numBands == 24){
      if ( sbrDecoderFilterbankCoefficientsInt != NULL)
        free(sbrDecoderFilterbankCoefficientsInt);
      
      sbrDecoderFilterbankCoefficientsInt = interpolPrototype(stdSbrDecoderFilterbankCoefficients,
                                                              640, 64, 24, 0);
      sbrDecoderFilterbankCoefficientsAna = sbrDecoderFilterbankCoefficientsInt;
      gain *= (4.0/3.0);
    } else {
      sbrDecoderFilterbankCoefficientsAna = stdSbrDecoderFilterbankCoefficients;
      if(numBands == 16) {
        gain *= 2.0;
      }
    }

#ifdef LOW_POWER_SBR
  for ( k=0; k<numBands; k++ ) {
    for ( l=0; l<2*numBands; l++ ) {
      M.real[k][l] = gain * cos((PI/(2*numBands))*(k+0.5)*(2*l-96));
      M.imag[k][l] = gain * sin((PI/(2*numBands))*(k+0.5)*(2*l-96));
    }
  }
#else /* LOW_POWER_SBR */
  for ( k=0; k<numBands; k++ ) {
    for ( l=0; l<2*numBands; l++ ) {
      M.real[k][l] = gain * cos((PI/(2*numBands))*(k+0.5)*(2*l-twiddleFac));
      M.imag[k][l] = gain * sin((PI/(2*numBands))*(k+0.5)*(2*l-twiddleFac));
    }
  }
#endif /* LOW_POWER_SBR */ 
#ifdef AAC_ELD
  }
#endif /* AAC_ELD */

}

void
CalculateSbrAnaFilterbank( float * timeSig,
                           float * Sr,
                           float * Si,
                           int channel,
                           int el,
                           int maxBand,
                           int numBands
#ifdef AAC_ELD
			   ,int ldsbr
#endif
			   )
{
  int i,j,k,l;
  double   Z[320];
  double   Y[64];
  double   realAccu, imagAccu;
  const    int stopPos = 10*numBands;
  const double* C = sbrDecoderFilterbankCoefficientsAna;

  /* shift input buffer */
  for ( i=stopPos-1; i>=numBands; i-- ) {
    X[el][channel][i] = X[el][channel][i-numBands];
  }

  /* add new samples to input buffer in reverse order */
  for ( i=numBands-1; i>=0; i-- ) {
    X[el][channel][i] = timeSig[stopPos-1-i];
  }

  /* window signal */
  for ( i=0; i<stopPos; i++ ) {
#ifdef AAC_ELD
  if (ldsbr)
    Z[i] = X[channel][i] * sbrDecoderFilterbankCoefficientsInt[i];
  else
#endif
    if(numBands == 16) {
      Z[i] = X[el][channel][i] * C[4*i];
    } else if (numBands == 24){
      Z[i] = X[el][channel][i] * C[1*i]; /* interpolated filter */
    } else {
      Z[i] = X[el][channel][i] * C[2*i];
    }
  }

  /* create array Y */
  for ( i=0; i<2*numBands; i++ ) {
    realAccu = 0.0;
    for ( j=0; j<5; j++ ) {
      realAccu = realAccu + Z[i + j * 2*numBands];
    }
    Y[i] = realAccu;
  }

#ifdef LOW_POWER_SBR
  /* Calculate 32 subband samples by matrixing */
  for ( k=0; k<numBands; k++ ) {
    realAccu = 0.0;
    imagAccu = 0.0;
    if (k<maxBand) {
      for ( l=0; l<2*numBands; l++ ) {
        realAccu = realAccu + Y[l] * M.real[k][l];
      }
    }
    Sr[k] = (float) realAccu;
    Si[k] = (float) imagAccu;
  }
#else
  /* Calculate 32 subband samples by matrixing */
  for ( k=0; k<numBands; k++ ) {
    realAccu = 0.0;
    imagAccu = 0.0;
    for ( l=0; l<2*numBands; l++ ) {
      realAccu = realAccu + Y[l] * M.real[k][l];
      imagAccu = imagAccu + Y[l] * M.imag[k][l];
    }
    Sr[k] = (float) realAccu;
    Si[k] = (float) imagAccu;
  }
#endif
}


void
InitSbrSynFilterbank(int bDownSampleSBR
#ifdef AAC_ELD
		     ,int ldsbr
#endif
		    )
{
  int l,k;

#ifdef LOW_POWER_SBR
  double gain = 2.0 / 64.0;
#else
  double gain = 1.0 / 64.0;
#endif

#ifdef AAC_ELD
  if (ldsbr) {
    sbrDecoderFilterbankCoefficientsSyn = ldSbrDecoderFilterbankCoefficients;
  } else {
#endif
    sbrDecoderFilterbankCoefficientsSyn = stdSbrDecoderFilterbankCoefficients;
#ifdef AAC_ELD
  }
#endif

#ifdef AAC_ELD
 if (ldsbr) {
  if (bDownSampleSBR){
    for ( l=0; l<64; l++ ) {
      for ( k=0; k<32; k++ ) {
        N.real[l][k] = gain * cos( PI / 64.0 * (k+0.5) * (2*l-31) );
#ifndef LOW_POWER_SBR
        N.imag[l][k] = gain * sin( PI / 64.0 * (k+0.5) * (2*l-31) );
#endif
      }
    }
  }
  else{
    for ( l=0; l<128; l++ ) {
      for ( k=0; k<64; k++ ) {
        N.real[l][k] = gain * cos( PI / 128.0 * (k+0.5) * (2*l-63) );
#ifndef LOW_POWER_SBR
        N.imag[l][k] = gain * sin( PI / 128.0 * (k+0.5) * (2*l-63) );
#endif
      }
    }
  }
 } else {
#endif /* AAC_ELD */

#ifdef LOW_POWER_SBR

  if (bDownSampleSBR){
    for ( l=0; l<64; l++ ) {
      for ( k=0; k<32; k++ ) {
        N.real[l][k] = gain * cos( PI / 64.0 * (k+0.5) * (2*l-32) );
      }
    }
  } else {
    for ( l=0; l<128; l++ ) {
      for ( k=0; k<64; k++ ) {
        N.real[l][k] = gain * cos( PI / 128.0 * (k+0.5) * (2*l-64) );
      }
    }
  }

#else /* LOW_POWER_SBR */

  if(bDownSampleSBR){
    for ( l=0; l<64; l++ ) {
      for ( k=0; k<32; k++ ) {
        N.real[l][k] = gain * cos( PI / 64.0 * (k+0.5) * (2*l-127.5) );
        N.imag[l][k] = gain * sin( PI / 64.0 * (k+0.5) * (2*l-127.5) );
      }
    }
  } else {
    for ( l=0; l<128; l++ ) {
      for ( k=0; k<64; k++ ) {
        N.real[l][k] = gain * cos( PI / 128.0 * (k+0.5) * (2*l-255) );
        N.imag[l][k] = gain * sin( PI / 128.0 * (k+0.5) * (2*l-255) );
      }
    }
  }
#endif /* LOW_POWER_SBR */
#ifdef AAC_ELD
 }
#endif /* AAC_ELD */
}


void
CalculateSbrSynFilterbank( float * Sr,
                           float * Si,
                           float * timeSig,
                           int bDownSampledSbr,
                           int channel,
                           int el
#ifdef AAC_ELD
			   ,int ldsbr
#endif
			    )
{
  int i,j,k,l;
  double   U[640];
  double   W[640];
  double   realAccu;
  const double* C = sbrDecoderFilterbankCoefficientsSyn;

  /* shift filterstates */
  if ( bDownSampledSbr){
    for ( l=639; l>=64; l-- ) {
      V[el][channel][l] = V[el][channel][l-64];
    }
  }
  else{
    for ( l=1279; l>=128; l-- ) {
      V[el][channel][l] = V[el][channel][l-128];
    }
  }

  /* create array V */
  if ( bDownSampledSbr){

    for ( l=0; l<64; l++ ) {
      realAccu = 0.0;
      for ( k=0; k<32; k++ ) {
        realAccu += Sr[k] * N.real[l][k];
#ifndef LOW_POWER_SBR
        realAccu -= Si[k] * N.imag[l][k];
#endif
      }
      V[el][channel][l] = realAccu;
    }
  }
  else{
    for ( l=0; l<128; l++ ) {
      realAccu = 0.0;
      for ( k=0; k<64; k++ ) {
        realAccu += Sr[k] * N.real[l][k];
#ifndef LOW_POWER_SBR
        realAccu -= Si[k] * N.imag[l][k];
#endif
      }
      V[el][channel][l] = realAccu;
    }
  }

  /* build the array U */
  if ( bDownSampledSbr){
    for ( i=0; i<5; i++ ) {
      for ( j=0; j<32; j++ ) {
        U[64*i+j]     = V[el][channel][128*i+j];
        U[64*i+32+j]  = V[el][channel][128*i+96+j];
      }
    }
  }
  else{
    for ( i=0; i<5; i++ ) {
      for ( j=0; j<64; j++ ) {
        U[128*i+j]     = V[el][channel][256*i+j];
        U[128*i+64+j]  = V[el][channel][256*i+192+j];
      }
    }
  }

  /* window signal */


  if ( bDownSampledSbr){
    for ( k=0; k<320; k++ ) {
#ifdef AAC_ELD
     if (ldsbr)
      W[k] = U[k] * sbrDecoderFilterbankCoefficientsInt[k];
     else
#endif
      W[k] = U[k] * C[2*k];
    }
  }
  else{
    for ( k=0; k<640; k++ ) {
      W[k] = U[k] * C[k];
    }
  }

  /* calculate output samples */
  if ( bDownSampledSbr){
    for ( i=0; i<32; i++ ) {
      realAccu = 0.0;
      for ( j=0; j<10; j++ ) {
        realAccu = realAccu + W[32*j+i];
      }
      timeSig[i] = (float) realAccu;
    }
  }
  else{
    for ( i=0; i<64; i++ ) {
      realAccu = 0.0;
      for ( j=0; j<10; j++ ) {
        realAccu = realAccu + W[64*j+i];
      }
      timeSig[i] = (float) realAccu;
    }
  }
}

void
CQMFanalysis( const float  *timeSig,
              float        *Sr,
              float        *Si,
              int           L, /* Filter bank size */
              double       *buffer,
              struct M2    *MM,
              const double *C)
{
  int i,j,k,l;
  double   Z[640];
  double   Y[128];
  double   realAccu, imagAccu;

  int N = (int)(640.0/64*L+0.5);

  /* shift input buffer */
  for ( i=N-1; i>=L; i-- ) {
    buffer[i] = buffer[i-L];
  }

  /* add new samples to input buffer in reverse order */
  for ( i=L-1; i>=0; i-- ) {
    buffer[i] = timeSig[L-1-i];
  }

  /* window signal */
  for ( i=0; i<N; i++ ) {
    Z[i] = buffer[i] * C[i];
  }

  /* create array Y */
  for ( i=0; i<2*L; i++ ) {
    realAccu = 0.0;
    for ( j=0; j<5; j++ ) {
      realAccu = realAccu + Z[i + j * 2*L];
    }
    Y[i] = realAccu;
  }

  /* Calculate L subband samples by matrixing */
  for ( k=0; k<L; k++ ) {
    realAccu = 0.0;
    imagAccu = 0.0;
    for ( l=0; l<2*L; l++ ) {
      realAccu = realAccu + Y[l] * MM->real[k][l];
      imagAccu = imagAccu + Y[l] * MM->imag[k][l];
    }
    Sr[k] = (float) realAccu;
    Si[k] = (float) imagAccu;
  }
}

void
RQMFsynthesis( const float  *Sr,
               float        *timeSig,
               int           L,
               double       *buffer,
               struct N     *NN,
               const double *C )
{
  int i,j,k,l;
  double   U[640];
  double   W[640];
  double   realAccu;

  int N = (int)(640.0/64*L+0.5);


  /* shift filterstates */
  for ( l=2*N-1; l>=2*L; l-- ) {
    buffer[l] = buffer[l-2*L];
    }

  /* create array V */
  for ( l=0; l<2*L; l++ ) {
    realAccu = 0.0;
    for ( k=0; k<L; k++ ) {
      realAccu += Sr[k] * NN->real[l][k];
    }
    buffer[l] = realAccu;
  }

  /* build the array U */
  for ( i=0; i<5; i++ ) {
    for ( j=0; j<L; j++ ) {
      U[2*L*i+j]   = buffer[4*L*i+j];
      U[2*L*i+L+j] = buffer[4*L*i+3*L+j];
    }
  }

  /* window signal */
  for ( k=0; k<N; k++ ) {
    W[k] = U[k] * C[k];
  }

  /* calculate output samples */
  for ( i=0; i<L; i++ ) {
    realAccu = 0.0;
    for ( j=0; j<10; j++ ) {
      realAccu = realAccu + W[L*j+i];
    }
    timeSig[i] = (float) realAccu;
  }
}
